import { html } from 'lit'

import '../ui/qrcg-box'
import '../ui/qrcg-input'
import '../ui/qrcg-button'
import '../ui/qrcg-form'
import '../ui/qrcg-link'
import { post } from '../core/api'
import { queryParam, isEmpty } from '../core/helpers'
import { QRCGAccountPage } from './qrcg-account-page'
import { t } from '../core/translate'

class QRCGPasswordReset extends QRCGAccountPage {
    static get properties() {
        return {
            data: {},

            errorMessage: {
                type: String,
                state: true,
            },
            success: {
                type: Boolean,
                state: true,
            },
        }
    }

    constructor() {
        super()

        this.data = {
            email: queryParam('email'),
            password: '',
            password_confirmation: '',
            token: queryParam('token'),
        }
    }

    async submitForm() {
        try {
            const { status } = await this.api.call(() =>
                post('reset-password', this.data)
            )

            this.success = this.isSuccess(status)

            if (!this.success) {
                this.errorMessage = this.statusError(status)
            }
        } catch (e) {
            console.error(e)
        }
    }

    isSuccess(status) {
        return status === 'passwords.reset'
    }

    statusError(status) {
        return {
            'passwords.user': t('User was not found'),
            'passwords.token': t('The reset link is expired.'),
            'passwords.throttled': t(
                'Password reset blocked, please try again later.'
            ),
        }[status]
    }

    renderInitial() {
        return html`
            ${!isEmpty(this.errorMessage)
                ? html`<p>${this.errorMessage}</p>`
                : ''}

            <qrcg-form>
                <qrcg-input
                    name="password"
                    type="password"
                    placeholder=${t`between 6 and 8 charachters`}
                >
                    ${t`Password`}
                </qrcg-input>

                <qrcg-input
                    name="password_confirmation"
                    type="password"
                    placeholder=${t`confirm your password`}
                >
                    ${t`Password confirmation`}
                </qrcg-input>

                <qrcg-button type="submit">${t`Submit`}</qrcg-button>
            </qrcg-form>
        `
    }

    renderSuccess() {
        return html`
            <p>
                ${t`Your password has been reset successfully. You can now login
                with your new password.`}
            </p>

            <qrcg-link href="/account/login">${t`Login`}</qrcg-link>
        `
    }

    renderContent() {
        return html`
            <h1>${t`Password reset`}</h1>
            ${this.success ? this.renderSuccess() : this.renderInitial()}
        `
    }
}

window.customElements.define('qrcg-reset-password', QRCGPasswordReset)
