import { get } from './api'

import { push } from './qrcg-router'

import { Config } from './qrcg-config'

import { get as getObjectProperty } from './helpers'

import { storeJson, loadStoredJson, isEmpty, queryParam } from './helpers'

const userKey = 'auth:user'

const tokenKey = 'auth:token'

export function userHomePage() {
    const user = loadUser()

    let homePage = ''

    homePage = getObjectProperty(user, `roles[0].home_page`)

    return homePage
}

export function loadUser() {
    return loadStoredJson(userKey)
}

function storeUser(user) {
    storeJson(user, userKey)
}

export function storeToken(token) {
    localStorage[tokenKey] = token
}

export function loadToken() {
    return localStorage[tokenKey]
}

export function login(user, token) {
    storeUser(user)
    storeToken(token)
}

export function hasToken() {
    return !isEmpty(loadToken())
}
export function loggedIn() {
    const user = loadUser()
    const token = loadToken()

    return !isEmpty(user) && !isEmpty(token)
}

export function verified() {
    const user = loadUser()

    if (isEmpty(user)) return false

    return !isEmpty(user.email_verified_at)
}

function clearLocalStorage() {
    delete localStorage[userKey]
    delete localStorage[tokenKey]
}

function onInvalidToken() {
    clearLocalStorage()

    document.dispatchEvent(new CustomEvent('auth:request-logout'))
}

function redirectToLoginPageAfterLogout(e) {
    let redirect = ''

    try {
        const eRedirect = getObjectProperty(e, 'detail.redirect')

        if (eRedirect) redirect = `?redirect=${eRedirect}`
    } catch (error) {
        redirect = ''
    }

    push(`/account/login${redirect}`)
}

function onLogoutRequested(e) {
    clearLocalStorage()

    const action = Config.get('app.after_logout_action')

    if (isEmpty(action) || action === 'redirect_to_login_page') {
        redirectToLoginPageAfterLogout(e)
        setTimeout(() => {
            location.reload()
        }, 50)
    } else if (action === 'redirect_to_homepage') {
        push('/')
    }
}

async function onAfterLogin(e) {
    const { user, token } = e.detail

    storeUser(user)

    storeToken(token)

    window.dispatchEvent(new CustomEvent('auth:local-user-ready'))

    if (queryParam('redirect')) {
        push(queryParam('redirect'))
    } else {
        push(userHomePage())
    }
}

export function isSuperAdmin() {
    const user = loadUser()

    if (isEmpty(user) || isEmpty(user.roles)) {
        return false
    }

    const superAdmin = user.roles.reduce(
        (superAdmin, role) => superAdmin || role.super_admin,
        false
    )

    return superAdmin
}

export function permitted(slug) {
    if (isEmpty(slug)) return true
    const user = loadUser()

    if (isEmpty(user) || isEmpty(user.roles)) {
        return false
    }

    const superAdmin = user.roles.reduce(
        (superAdmin, role) => superAdmin || role.super_admin,
        false
    )

    if (superAdmin) {
        return true
    }

    if (!verified()) return false

    const permissions = user.roles
        .map((role) => role.permissions)
        .reduce((perms, arr) => perms.concat(arr), [])

    return !!permissions.find((p) => p.slug === slug)
}

function main() {
    window.addEventListener('qrcg-login:after-login', onAfterLogin)

    if (
        document.readyState === 'complete' ||
        document.readyState === 'loaded' ||
        document.readyState === 'interactive'
    ) {
        // document has at least been parsed
        onDocumentReady()
    } else {
        window.addEventListener('DOMContentLoaded', onDocumentReady)
    }

    window.addEventListener('storage', onWindowStorageChanged)

    window.addEventListener('auth:invalid-token', onInvalidToken)

    window.addEventListener('auth:request-logout', onLogoutRequested)
}

export function logout() {
    onLogoutRequested()
}

async function onWindowStorageChanged() {
    await new Promise((resolve) => setTimeout(resolve, 0))

    // This behavior is buggy and error prune, disabling ...
    // validateCurrentToken()
}

function onDocumentReady() {
    validateCurrentToken()
}

export async function validateCurrentToken() {
    const token = loadToken()

    if (!token) {
        window.dispatchEvent(new CustomEvent('auth:invalid-token'))
        return
    }

    try {
        const { response } = await get('myself')

        const user = await response.clone().json()

        storeUser(user)

        window.dispatchEvent(new CustomEvent('auth:local-user-ready'))
    } catch (error) {
        window.dispatchEvent(new CustomEvent('auth:invalid-token'))
        // console.error(error)
    }
}

main()
