import { html, css, LitElement } from 'lit'

import { mdiBackburger } from '@mdi/js'

import { classMap } from 'lit/directives/class-map.js'
import { t } from '../core/translate'
import { logout } from '../core/auth'

export class QRCGDashboardHeader extends LitElement {
    static get styles() {
        return [
            css`
                :host {
                    display: flex;
                    background-color: blue;
                    height: var(--dashboard-header-height);
                    background-color: white;
                    box-shadow: 2px 2px 2px rgba(0, 0, 0, 0.1);
                    padding: 0 1rem;
                    align-items: center;
                    position: sticky;
                    top: 0;
                    z-index: 10;
                }

                .sidebar-toggle::part(button) {
                    padding: 0.3rem;
                    min-width: fit-content;
                    border-radius: 0.2rem;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }

                .push {
                    flex: 1;
                }

                .logout {
                    font-weight: bold;
                    text-transform: uppercase;
                    letter-spacing: 1px;
                    font-size: 0.8rem;
                    text-decoration: none;
                    color: var(--primary-0);
                    cursor: pointer;
                }

                .logout:hover {
                    text-decoration: underline;
                }

                .sidebar-toggle.closed {
                    transform: scaleX(-1);
                }

                .sidebar-toggle {
                    display: block;
                }

                .sidebar-toggle::part(button) {
                    border-radius: 50%;
                    width: 2rem;
                    height: 2rem;
                }

                .sidebar-toggle qrcg-icon {
                    width: 1rem;
                    height: 1rem;
                }

                @media (min-width: 900px) {
                    .sidebar-toggle {
                        display: none;
                    }
                }

                qrcg-app-logo {
                    margin-right: 1rem;
                }
            `,
        ]
    }

    static get properties() {
        return {
            ...super.properties,
            sideBarClosed: { state: true },
        }
    }

    constructor() {
        super()
        this.setSidebarClosed = this.setSidebarClosed.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()

        document.addEventListener(
            'qrcg-dashboard-sidebar:status-changed',
            this.setSidebarClosed
        )

        this.constructor._isRendered = true
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        document.removeEventListener(
            'qrcg-dashboard-sidebar:status-changed',
            this.setSidebarClosed
        )

        this.constructor._isRendered = false
    }

    setSidebarClosed(e) {
        this.sideBarClosed = e.detail.closed
    }

    toggleSidebar() {
        document.dispatchEvent(
            new CustomEvent('qrcg-dashboard-sidebar:request-toggle')
        )
    }

    static get isRendered() {
        return this._isRendered
    }

    onLogout() {
        logout()
    }

    render() {
        return html`
            <qrcg-app-logo variation="inverse" href="/"></qrcg-app-logo>
            <qrcg-button
                class="sidebar-toggle ${classMap({
                    closed: this.sideBarClosed,
                })}"
                @click=${this.toggleSidebar}
            >
                <qrcg-icon mdi-icon=${mdiBackburger}></qrcg-icon>
            </qrcg-button>
            <div class="push"></div>
            <div class="logout" @click=${this.onLogout}>${t`Logout`}</div>
        `
    }
}

window.customElements.define('qrcg-dashboard-header', QRCGDashboardHeader)
