import { LitElement, html, css, unsafeCSS } from 'lit'

import { classMap } from 'lit/directives/class-map.js'
import { QrcgBuyToolbar } from '../common/qrcg-buy-toolbar'

import { isSuperAdmin, permitted } from '../core/auth'
import { Config } from '../core/qrcg-config'

import '../core/qrcg-route'
import { t } from '../core/translate'

import './qrcg-dashboard-sidebar-account'

class QRCGDashboardSidebar extends LitElement {
    static TRANSITION_DURATION = '.2s'

    static get styles() {
        return css`
            :host {
                --transition-duration: 0.2s;
                --sidebar-account-height: 8rem;
                display: none;
                background-color: var(--primary-0);
                box-shadow: 2px 2px 2px rgba(0, 0, 0, 0.1);
                min-height: calc(100vh - var(--dashboard-header-height));
                width: var(--dashboard-sidebar-width, 15rem);
                overflow: hidden;
                transition: width ${unsafeCSS(this.TRANSITION_DURATION)}
                    ease-in-out;
                user-select: none;
                -webkit-user-select: none;
                touch-action: manipulation;
                z-index: 20000;
            }

            @media (min-width: 800px) {
                :host {
                    display: block;
                }
            }

            :host(.updated) {
                display: block;
            }

            :host([mode='fixed']) {
                position: fixed;
                top: var(--dashboard-header-height);
                bottom: 0;
                left: 0;
            }

            :host([mode='relative']) {
                position: sticky;
                top: var(--dashboard-header-height);
                height: min-content;
                min-height: calc(100vh - var(--dashboard-header-height));
            }

            :host([closed]) {
                width: 0;
            }

            @media (min-width: 900px) {
                :host([closed]) {
                    width: 1.5rem;
                }
            }

            qrcg-dashboard-sidebar-account {
                height: var(--sidebar-account-height);
                position: absolute;
                left: 0;
                width: var(--dashboard-sidebar-width, 15rem);
                bottom: 0;

                transition: transform ${unsafeCSS(this.TRANSITION_DURATION)}
                    ease-in-out;
            }

            .content {
                position: absolute;
                width: var(--dashboard-sidebar-width, 15rem);
                top: 0;
                left: 0;
                bottom: var(--sidebar-account-height);
                transition: transform ${unsafeCSS(this.TRANSITION_DURATION)}
                    ease-in-out;
                padding: 0.5rem;
                box-sizing: border-box;
                overflow: scroll;
                scrollbar-width: none;
            }

            .content::-webkit-scrollbar {
                display: none;
            }

            :host([closed]) .content,
            :host([closed]) qrcg-dashboard-sidebar-account {
                transform: translateX(-100%);
            }

            .item {
                display: block;
                padding: 0.6rem 1rem;
                margin: 0.5rem 0;
                color: white;
                font-size: 0.9rem;
                border-radius: 0.3rem;
                cursor: pointer;
                text-decoration: none;
                -webkit-tap-highlight-color: transparent;
                transition: 0.1s ease;
                -webkit-touch-callout: none;
            }

            .item.active {
                background-color: var(--gray-0);
                color: var(--primary-0);
            }

            @media (hover) {
                .item:hover {
                    background-color: var(--gray-0);
                    color: var(--primary-0);
                }
            }

            label {
                font-weight: bold;
                display: block;
                color: white;
                font-size: 0.65rem;
                margin: 2rem 0 0.5rem 1rem;
                letter-spacing: 0.05rem;
                text-transform: uppercase;
            }
        `
    }

    static get properties() {
        return {
            closed: { type: Boolean, reflect: true },
            groups: { state: true },
            bodyStyleId: {},
            overlayId: {},
            overlayStyleId: {},
            overlayAnimation: {},
            mode: { type: String, reflect: true },
        }
    }

    constructor() {
        super()

        this.onLocationChanged = this.onLocationChanged.bind(this)

        this.onOverlayClick = this.onOverlayClick.bind(this)
        this.requestClose = this.requestClose.bind(this)
        this.requestOpen = this.requestOpen.bind(this)
        this.onStatusChanged = this.onStatusChanged.bind(this)

        this.onLargeScreenMediaQueryChange =
            this.onLargeScreenMediaQueryChange.bind(this)

        this.groups = [
            {
                name: t('QR Codes'),
                items: [
                    {
                        label: t('All'),
                        link: '/dashboard/qrcodes',
                        permission: 'qrcode.list',
                    },
                    {
                        label: t('Archived'),
                        link: '/dashboard/qrcodes?archived=true',
                        permission: 'qrcode.list',
                    },
                ],
            },
            {
                name: t('Users'),
                items: [
                    {
                        label: t('All users'),
                        link: '/dashboard/users',
                        permission: 'user.list-all',
                    },
                    {
                        label: t('Paying users'),
                        link: '/dashboard/users?paying=true',
                        permission: 'user.list-all',
                    },
                    {
                        label: t('Non paying users'),
                        link: '/dashboard/users?paying=false',
                        permission: 'user.list-all',
                    },
                ],
            },
            {
                name: t('Finance'),
                items: [
                    {
                        label: t('Plans'),
                        link: '/dashboard/subscription-plans',
                        permission: 'subscription-plan.list-all',
                    },
                    {
                        label: t('Subscriptions'),
                        link: '/dashboard/subscriptions',
                        permission: 'subscription.list-all',
                    },
                    {
                        label: t('Transactions'),
                        link: '/dashboard/transactions',
                        permission: 'transaction.list-all',
                    },
                    {
                        label: t('Payment gateways'),
                        link: '/dashboard/payment-gateways',
                        permission: 'payment-gateway.list-all',
                    },
                    {
                        label: t`Currencies`,
                        link: '/dashboard/currencies',
                        permission: 'currency.list-all',
                    },
                ],
            },
            {
                name: t('Content'),
                items: [
                    {
                        label: t('Blog posts'),
                        link: '/dashboard/blog-posts',
                        permission: 'blog-post.list-all',
                    },
                    {
                        label: t('Content blocks'),
                        link: '/dashboard/content-blocks',
                        permission: 'content-block.list-all',
                    },
                    {
                        label: t`Translations`,
                        link: '/dashboard/translations',
                        permission: 'translations.list-all',
                    },
                    {
                        label: t('Custom Code'),
                        link: '/dashboard/custom-codes',
                        permission: 'custom-code.list-all',
                    },
                    {
                        label: t('Pages'),
                        link: '/dashboard/pages',
                        permission: 'pages.list-all',
                    },
                ],
            },
            {
                name: t('Contacts'),
                items: [
                    {
                        label: t('Contact form'),
                        link: '/dashboard/contacts',
                        permission: 'contact.list-all',
                    },
                ],
            },
            {
                name: t('System'),
                items: [
                    {
                        label: t('Status'),
                        link: '/dashboard/system/status',
                        permission: 'system.status',
                    },
                    {
                        label: t('Settings'),
                        link: '/dashboard/system/settings',
                        permission: 'system.settings',
                    },
                ],
            },
        ]

        if (!isSuperAdmin()) {
            this.groups = [
                ...this.groups,
                ...(Config.get('app.dashboard-client-menu') ?? []),
            ]
        }

        this.bodyStyleId = 'qrcg-dashboard-sidebar-body-style'

        this.overlayId = 'qrcg-dashboard-sidebar-overlay'

        this.overlayStyleId = 'qrcg-dashboard-sidebar-overlay-style'

        this.overlayAnimation = 'qrcg-dashboard-sidebar-animation'

        this.mode = 'fixed|relative'.split('|')[1]

        this.largeScreenMediaQuery = window.matchMedia('(min-width: 800px)')
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener(
            'qrcg-dashboard-sidebar:status-changed',
            this.onStatusChanged
        )

        this.addEventListener('mouseenter', this.onMouseEnter)

        this.addEventListener('mouseleave', this.onMouseLeave)

        this.largeScreenMediaQuery.addEventListener(
            'change',
            this.onLargeScreenMediaQueryChange
        )

        window.addEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        window.addEventListener('resize', this.adjustVisualViewportRendering)

        window.addEventListener('scroll', this.adjustVisualViewportRendering)
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        this.removeEventListener(
            'qrcg-dashboard-sidebar:status-changed',
            this.onStatusChanged
        )

        this.removeEventListener('mouseenter', this.onMouseEnter)

        this.removeEventListener('mouseleave', this.onMouseLeave)

        this.largeScreenMediaQuery.removeEventListener(
            'change',
            this.onLargeScreenMediaQueryChange
        )

        window.removeEventListener(
            'qrcg-router:location-changed',
            this.onLocationChanged
        )

        this.removeOverlay()

        this.removeBodyStyles()

        window.removeEventListener('resize', this.adjustVisualViewportRendering)
        window.removeEventListener('scroll', this.adjustVisualViewportRendering)
    }

    onMouseEnter() {
        document.dispatchEvent(
            new CustomEvent('qrcg-dashboard-sidebar:mouseenter')
        )
    }

    shouldRenderFrontendLinks() {
        const config = Config.get('app.frontend_links')

        return !config || config === 'enabled'
    }

    onMouseLeave() {
        document.dispatchEvent(
            new CustomEvent('qrcg-dashboard-sidebar:mouseleave')
        )
    }

    onLocationChanged() {
        this.requestUpdate()
    }

    firstUpdated() {
        this.onLargeScreenMediaQueryChange()

        setTimeout(() => {
            this.classList.add('updated')
        }, 0)
    }

    adjustVisualViewportRendering = () => {
        const content = this.shadowRoot.querySelector('.content')

        const account = this.renderRoot.querySelector(
            'qrcg-dashboard-sidebar-account'
        )

        const toolbarHeight = QrcgBuyToolbar.toolbarHeight

        let toolbarMargin = Math.max(
            0,
            toolbarHeight - document.documentElement.scrollTop
        )

        if (this.mode === 'fixed') {
            this.style.top = `calc(var(--dashboard-header-height) + ${toolbarMargin}px)`
        }

        const bottomMargin = window.visualViewport.height - toolbarMargin

        account.style.bottom = `calc(100vh - ${bottomMargin}px)`

        content.style.bottom = `calc(var(--sidebar-account-height) + (100vh - ${bottomMargin}px))`
    }

    updated(changed) {
        if (changed.has('closed')) {
            this.dispatchEvent(
                new CustomEvent('qrcg-dashboard-sidebar:status-changed', {
                    bubbles: true,
                    composed: true,
                    detail: {
                        closed: this.closed,
                    },
                })
            )
        }

        if (changed.has('mode')) {
            this.onModeChange()
        }

        setTimeout(() => this.adjustVisualViewportRendering(), 0)
    }

    createBodyStyles() {
        const style = document.createElement('style')
        style.innerHTML = `body { position: fixed; top: 0; left: 0; bottom: 0; right: 0; overflow: hidden; }`
        style.id = this.bodyStyleId
        document.head.appendChild(style)
    }

    removeBodyStyles() {
        document.getElementById(this.bodyStyleId)?.remove()
    }

    createOverlay() {
        const style = document.createElement('style')

        style.innerHTML = `#${this.overlayId} { 
            position: fixed;
            top: 0;
            bottom: 0;
            left: 0;
            right: 0;
            background-color: black;
            opacity: 0.8;
            animation: ${this.overlayAnimation} ${this.constructor.TRANSITION_DURATION} ease-in both;
        }

        @keyframes ${this.overlayAnimation} {
            from {
                opacity: 0;
            }

            to {
                opacity: 0.5;
            }
        }
        `

        style.id = this.overlayStyleId

        document.head.appendChild(style)

        const div = document.createElement('div')

        div.id = this.overlayId

        div.addEventListener('click', this.onOverlayClick)

        this.overlay = div

        this.overlayStyle = style

        this.parentElement.appendChild(this.overlay)
        this.parentElement.appendChild(this.overlayStyle)
    }

    removeOverlay() {
        this.overlay?.remove()
        this.overlayStyle?.remove()
    }

    onOverlayClick() {
        this.requestClose()
    }

    requestClose() {
        document.dispatchEvent(
            new CustomEvent('qrcg-dashboard-sidebar:request-close')
        )
    }

    requestOpen() {
        document.dispatchEvent(
            new CustomEvent('qrcg-dashboard-sidebar:request-open')
        )
    }

    onStatusChanged(e) {
        if (this.mode === 'relative') {
            return
        }

        if (e.detail.closed) {
            this.removeBodyStyles()
            this.removeOverlay()
        } else {
            this.createBodyStyles()
            this.createOverlay()
        }
    }

    onModeChange() {
        if (this.mode === 'relative') {
            this.removeBodyStyles()
            this.removeOverlay()
        }
    }

    onLargeScreenMediaQueryChange() {
        if (this.largeScreenMediaQuery.matches) {
            this.mode = 'relative'

            this.requestOpen()
        } else {
            this.mode = 'fixed'
            this.requestClose()
        }
    }

    isActive(item) {
        const itemSearchPart = item.link.match(/\?/)
            ? item.link.replace(/.*(\?.*)/, (_, $1) => {
                  return $1
              })
            : ''

        let itemSearch = new URLSearchParams(itemSearchPart)

        const itemPath = item.link.replace(itemSearchPart, '')

        let windowSearch = window.location.search

        windowSearch = new URLSearchParams(windowSearch)

        const itemSearchMatched = Array.from(itemSearch.keys()).reduce(
            (matched, key) => {
                return matched && itemSearch.get(key) === windowSearch.get(key)
            },
            true
        )

        const ignoredKeys = ['page']

        const windowHasExtraParams = Array.from(windowSearch.keys()).reduce(
            (prev, key) => {
                const keyFoundInItemSearch = Array.from(itemSearch.keys()).find(
                    (iKey) => key === iKey
                )

                const keyIsIgnored = ignoredKeys.find((iKey) => iKey === key)

                if (keyIsIgnored) {
                    return prev
                }

                if (keyFoundInItemSearch) {
                    return prev
                } else {
                    return true
                }
            },
            false
        )

        const result =
            itemPath === window.location.pathname &&
            itemSearchMatched &&
            !windowHasExtraParams

        return result
    }

    renderItem(item) {
        if (!permitted(item.permission)) {
            return null
        }

        return html`
            <a
                class="item ${classMap({
                    active: this.isActive(item),
                })}"
                href="${item.link}"
                target=${item.target || '_self'}
            >
                ${item.label}
            </a>
        `
    }

    renderGroup(group) {
        const shouldRenderGroup = group.items.reduce(
            (anyPermitted, item) => anyPermitted || permitted(item.permission),
            false
        )

        if (!shouldRenderGroup) return null

        return html`
            <label>${group.name}</label>
            ${group.items.map((item) => this.renderItem(item))}
        `
    }

    renderGroups() {
        return this.groups.map((group) => this.renderGroup(group))
    }

    render() {
        return html`
            <div class="content">
                ${this.renderGroups()}
                <div class="content-bottom-gap"></div>
            </div>
            <qrcg-dashboard-sidebar-account></qrcg-dashboard-sidebar-account>
        `
    }
}

window.customElements.define('qrcg-dashboard-sidebar', QRCGDashboardSidebar)
