import { LitElement, css } from 'lit'

import { QRCGCheckoutController } from '../checkout/qrcg-checkout-controller'

import { isEmpty } from '../core/helpers'

import { QRCGApiConsumer } from '../core/qrcg-api-consumer'

import { t } from '../core/translate'

import { mdiCheck } from '@mdi/js'

import '../ui/qrcg-file-input/index'

import '../ui/qrcg-icon'

import { push } from '../core/qrcg-router'

import { unsafeStatic, html } from 'lit/static-html.js'

export class QrcgOfflinePaymentButton extends LitElement {
    checkoutController = new QRCGCheckoutController(this)

    static styles = [
        css`
            :host {
                display: flex;
                flex-direction: column;
            }

            .loader-container {
                display: flex;
                justify-content: center;
                align-items: center;
            }

            qrcg-button {
                margin-top: 1rem;
            }

            qrcg-file-input {
                margin: 0.2rem;
                margin-top: 1rem;
                max-width: initial;
            }

            .subscribe-instructions {
                display: flex;
                align-items: center;
                justify-content: center;
                margin-top: 1rem;
                animation: fade-in 0.5s ease-in both;
                background-color: var(--gray-0);
                padding: 1rem;
                border-radius: 0.5rem;
                font-size: 0.8rem;

                line-height: 1.6;
            }

            qrcg-form-comment {
                border-radius: 0.5rem;
            }

            .success-icon {
                display: none;
                width: 1.5rem;
                height: 1.5rem;
                color: var(--success-0);
                margin-right: 1rem;
            }

            .customer-instructions p,
            .customer-instructions ul {
                margin: 0;
            }

            .customer-instructions a {
                color: var(--primary-0);
                text-decoration: none;
            }

            @media (min-width: 800px) {
                .success-icon {
                    display: block;
                }
            }

            @keyframes fade-in {
                from {
                    opacity: 0;
                    margin-top: 0;
                }

                to {
                    opacity: 1;
                    margin-top: 1rem;
                }
            }
        `,
    ]

    static get properties() {
        return {
            offlinePaymentGateway: {},
            planId: {},
            redirectLoading: { type: Boolean },
            paymentProofId: {},
            paymentProofSubscribeLoading: { type: Boolean },
        }
    }

    constructor() {
        super()

        this.api = QRCGApiConsumer.instance({
            host: this,
            baseRoute: '',
            bindEvents: true,
        })
    }

    connectedCallback() {
        super.connectedCallback()

        this.fetchOfflinePaymentGateway()

        this.planId = this.checkoutController.loadPlanId()
    }

    disconnectedCallback() {
        super.disconnectedCallback()
    }

    async fetchOfflinePaymentGateway() {
        const response = await this.api.getRoute('payment-gateways')

        this.offlinePaymentGateway = response.data.find(
            (p) => p.slug === 'offline-payment-gateway'
        )
    }

    renderLoader() {
        return html`
            <div class="loader-container">
                <qrcg-loader></qrcg-loader>
            </div>
        `
    }

    renderInstructions() {
        return html`
            <qrcg-form-comment label="">
                <div class="customer-instructions">
                    ${unsafeStatic(
                        this.offlinePaymentGateway.customer_instructions_html
                    )}
                </div>
            </qrcg-form-comment>
        `
    }

    async subscribe() {
        return await this.api.post('subscriptions/subscribe', {
            subscription_plan_id: this.planId,
        })
    }

    async onRedirectButtonClick() {
        this.redirectLoading = true

        await this.subscribe()

        window.location = this.offlinePaymentGateway.redirect_url
    }

    async onPaymentProofSubscibeClick() {
        this.paymentProofSubscribeLoading = true

        const subscription = await this.subscribe()

        const data = {
            subscription_id: subscription.id,
            payment_proof_id: this.paymentProofId,
        }

        await this.api.post('transactions/offline-transaction', data)

        push('/checkout/offline-payment-received')
    }

    renderRedirectLink() {
        const link = this.offlinePaymentGateway.redirect_url

        if (isEmpty(link)) return null

        return html`
            <qrcg-button
                @click=${this.onRedirectButtonClick}
                class="offline-pay-redirect-button"
                ?loading=${this.redirectLoading}
            >
                ${t`Pay Now`}
            </qrcg-button>
        `
    }

    onProofOfPaymentFileInput(e) {
        const file = e.detail.value

        this.paymentProofId = file
    }

    renderAttachmentInput() {
        if (!isEmpty(this.offlinePaymentGateway.redirect_url)) return null

        return html`
            <qrcg-file-input
                name="proof_of_payment"
                upload-endpoint=${'transactions/upload-proof-of-payment'}
                @on-input=${this.onProofOfPaymentFileInput}
                ?disabled=${this.paymentProofSubscribeLoading}
            >
                ${t`Proof Of Payment`}
            </qrcg-file-input>

            ${!isEmpty(this.paymentProofId)
                ? html`
                      <div class="subscribe-instructions">
                          <qrcg-icon
                              class="success-icon"
                              mdi-icon=${mdiCheck}
                          ></qrcg-icon>
                          ${t`Uploaded successfully! Click on Subscribe Now, and wait for our confirmation email.`}
                      </div>
                  `
                : null}

            <qrcg-button
                ?disabled=${isEmpty(this.paymentProofId)}
                ?loading=${this.paymentProofSubscribeLoading}
                @click=${this.onPaymentProofSubscibeClick}
            >
                ${t`Send Transaction Details`}
            </qrcg-button>
        `
    }

    render() {
        if (!this.offlinePaymentGateway) {
            return this.renderLoader()
        }

        return html`
            ${this.renderInstructions()}
            <!-- new line -->

            ${this.renderRedirectLink()}

            <!-- new line -->
            ${this.renderAttachmentInput()}
        `
    }
}

customElements.define('qrcg-offline-payment-button', QrcgOfflinePaymentButton)
