import { LitElement, html, css } from 'lit'

import '../ui/qrcg-pricing-table'

import { QRCGApiConsumer } from '../core/qrcg-api-consumer'

import { QRCGCheckoutController } from '../checkout/qrcg-checkout-controller'

import { push } from '../core/qrcg-router'
import { loggedIn } from '../core/auth'
import { showToast } from '../ui/qrcg-toast'

import '../ui/qrcg-heading'
import { t } from '../core/translate'

export class QrcgPricingPlans extends LitElement {
    api = new QRCGApiConsumer(this, 'subscription-plans')

    checkout = new QRCGCheckoutController(this)

    static styles = [
        css`
            :host {
                display: block;
            }

            .description {
                color: var(--gray-2);
            }

            header {
                text-align: center;
            }
        `,
    ]

    static get properties() {
        return { plans: { type: Array } }
    }

    constructor() {
        super()

        this.plans = []

        this.onBuyClick = this.onBuyClick.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()

        setTimeout(() => {
            this.api.search({
                is_hidden: false,
            })
        }, 0)

        this.addEventListener('api:success', this.onApiSuccess)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('api:success', this.onApiSuccess)
    }

    getPlans(list) {
        return list.sort((a, b) => a.monthly_price - b.monthly_price)
    }

    onApiSuccess = async (e) => {
        const jsonResponse = await e.detail.rawResponse.json()

        this.plans = this.getPlans(jsonResponse.data)
    }

    onBuyClick(e) {
        if (!loggedIn()) {
            push('/account/login')

            showToast(
                t`Please sign in or create a new account to place your order`
            )

            return
        }

        this.checkout.savePlanId(e.detail.planId)

        push('/checkout')
    }

    render() {
        return html`
            <header>
                <qrcg-h2>${t`Plans & Pricing`}</qrcg-h2>
                <qrcg-p class="description">
                    ${t`Simple pricing plan, no hidden fees.`}
                </qrcg-p>
            </header>
            <qrcg-pricing-table
                .plans=${this.plans}
                @on-buy-click=${this.onBuyClick}
            ></qrcg-pricing-table>
        `
    }
}
customElements.define('qrcg-pricing-plans', QrcgPricingPlans)
