import { LitElement, css, html } from 'lit'

import {
    getScrollbarWidth,
    isEmpty,
    isFunction,
    remToPx,
    studlyCase,
    titleCase,
} from '../core/helpers'

import { QRCodePreviewUrlBuilder } from './qrcg-qrcode-preview-url-builder'

import '../ui/qrcg-color-picker'

import '../ui/qrcg-select'

import '../ui/qrcg-balloon-selector'

import '../ui/qrcg-file-input/index'

import '../ui/qrcg-range-input'

import '../ui/qrcg-logo-picker'

import '../ui/qrcg-img-selector'

import { t } from '../core/translate'

import '../common/qrcg-font-picker'

import '../ui/qrcg-gradient-input/qrcg-gradient-input'

import { QRCGDashboardLayout } from '../dashboard/qrcg-dashboard-layout'

import { QRCGDashboardHeader } from '../dashboard/qrcg-dashboard-header'

import { qrShapes } from '../models/qr-shapes'

import { featureAllowed } from '../core/subscription/logic'

import '../common/qrcg-sticker-text-input'

import { advancedShapes } from '../models/advanced-shapes'

export class QRCGQRCodeDesigner extends LitElement {
    urlBuilder = new QRCodePreviewUrlBuilder(this)

    static get styles() {
        return css`
            :host {
                display: block;
            }

            .row {
                display: flex;
                justify-content: flex-start;
                flex-direction: column;
            }

            @media (min-width: 70rem) {
                .row {
                    flex-direction: row;
                }
            }

            qrcg-qrcode-image {
                width: 15rem;
                margin: auto;
                border: solid 0.5rem var(--gray-0);
            }

            .preview-image {
                order: 1;
                width: 100%;
                margin: auto;
                /* max-width: 20rem; */
                height: fit-content;
                margin-bottom: 1rem;
                padding-bottom: 1rem;
            }

            @media (min-width: 70rem) {
                .preview-image {
                    margin: 0;
                    width: 25%;
                    order: 3;
                    max-width: unset;
                    box-sizing: border-box;
                }

                qrcg-qrcode-image {
                    width: unset;
                }
            }

            .input-grid {
                display: grid;
                align-items: center;
                box-sizing: border-box;
                padding: 0 0.5rem;
                grid-gap: 1rem;
                /* width: 75%; */
                order: 2;
            }

            qrcg-range-input {
                max-width: 30rem;
            }

            @media (min-width: 70rem) {
                .input-grid {
                    grid-template-columns: 12rem 1fr;
                    width: 75%;
                    padding: 0;
                    padding-right: 1rem;
                }
            }

            label {
                font-weight: bold;
                user-select: none;
                -webkit-user-select: none;
            }

            .warning {
                background-color: var(--warning-0);
                font-size: 0.8rem;
                margin-top: 1rem;
                padding: 0.5rem;
                line-height: 1.4;
                color: black;
            }

            [name='shape']::part(image) {
                padding: 0.25rem;
            }

            [name='text'] {
                max-width: 20rem;
            }
        `
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('on-input', this._onInput)

        this.updatePreview()

        window.addEventListener('scroll', this.onScroll)

        window.addEventListener('resize', this.onResize)

        window.scrollTo({ top: 0, behavior: 'auto' })
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('on-input', this._onInput)

        window.removeEventListener('scroll', this.onScroll)
        window.removeEventListener('resize', this.onResize)
    }

    static get properties() {
        return {
            previewUrl: { type: String },
            data: {},
            type: {},
            design: {},
            remoteRecord: {},
            stickTopRem: { attribute: 'stick-top-rem' },
            enableLargePreview: {
                type: Boolean,
                attribute: 'enable-large-preview',
            },
        }
    }

    constructor() {
        super()
        this.stickTopRem = 5
    }

    getQrCodeData() {
        return this.data
    }

    getQrCodeType() {
        return this.type
    }

    getQrCodeColor() {
        return this.design.codeColor
    }

    getQrCodeDesign() {
        return this.design
    }

    getQrCodeRemoteRecord() {
        return this.remoteRecord
    }

    onResize = () => {
        this.adjustPreviewImage()
    }

    adjustPreviewImage() {
        const image = this.shadowRoot.querySelector('.preview-image')

        const media = matchMedia('(min-width: 70rem)')

        if (!this.previewImageRect) {
            this.previewImageRect = image.getBoundingClientRect()
        }

        const topMargin = remToPx(this.stickTopRem)

        const offsetTop = this.previewImageRect.top - topMargin

        if (
            window.scrollY > offsetTop &&
            media.matches &&
            this.scrollingInsideDesigner()
        ) {
            image.style.position = 'fixed'
            image.style.zIndex = '100000'
            image.style.top = `${topMargin}px`

            image.style.right = `calc(100vw - ${
                this.previewImageRect.right + this.scrollbarWidth()
            }px)`
            image.style.backgroundColor = 'white'
            image.style.width = this.imageWidth + 'px'
        } else {
            image.style = ''
            this.imageWidth = image.clientWidth
        }

        if (!media.matches) {
            this.adjustPreviewImageOnMobile()
        }
    }

    adjustPreviewImageOnMobile() {
        const image = this.shadowRoot.querySelector('.preview-image')

        if (!this.previewImageRect) {
            this.previewImageRect = image.getBoundingClientRect()
        }

        const headerHeight = QRCGDashboardHeader.isRendered
            ? remToPx(
                  getComputedStyle(this)
                      .getPropertyValue('--dashboard-header-height')
                      .replace('rem', '')
              )
            : 0

        const offsetTop = this.previewImageRect.top - remToPx(1) - headerHeight

        if (
            window.scrollY > offsetTop &&
            this.scrollingInsideDesignerMobile()
        ) {
            image.style.position = 'fixed'

            image.style.zIndex = 100000

            image.style.paddingTop = '1rem'

            image.style.top = `${headerHeight}px`

            image.style.left = '2rem'

            if (!QRCGDashboardLayout.sidebarClosed) {
                image.style.left = '17rem'
            }

            image.style.right = '2rem'

            image.style.margin = ''

            image.style.width = 'initial'

            image.style.backgroundColor = 'white'

            this.setImagePlaceholderVisible(true)
        } else {
            this.setImagePlaceholderVisible(false)
        }
    }

    scrollbarWidth() {
        const scrollbarIsVisible =
            getComputedStyle(document.body, ':-webkit-scrollbar')?.display !==
            'none'

        if (scrollbarIsVisible) {
            return getScrollbarWidth()
        }

        return 0
    }

    scrollingInsideDesignerMobile() {
        const image = this.shadowRoot.querySelector('.preview-image')

        let imageHeight = image.getBoundingClientRect().height

        const isInside = this.getBoundingClientRect().bottom > imageHeight * 2

        return isInside
    }

    scrollingInsideDesigner() {
        const image = this.shadowRoot.querySelector('.preview-image')

        let imageHeight = image.getBoundingClientRect().height

        const isInside = this.getBoundingClientRect().bottom > imageHeight

        return isInside
    }

    setImagePlaceholderVisible(visible) {
        const image = this.shadowRoot.querySelector('.preview-image')

        if (!this.imagePlaceholder) {
            const style = getComputedStyle(image)

            this.imagePlaceholder = document.createElement('div')

            this.imagePlaceholder.style.height = style.height

            this.imagePlaceholder.style.margin = style.margin

            this.imagePlaceholder.style.padding = style.padding

            image.parentNode.insertBefore(this.imagePlaceholder, image)
        }

        if (visible) {
            this.imagePlaceholder.style.display = 'block'
        } else {
            this.imagePlaceholder.style.display = 'none'
        }
    }

    onScroll = () => {
        this.adjustPreviewImage()
    }

    updated() {
        this.renderRoot
            .querySelectorAll(`[name]`)
            .forEach((input) => (input.value = this.design[input.name]))
    }

    _onInput = (e) => {
        if (
            !isEmpty(e.detail.name) &&
            e.detail.name !== 'qrcg-qrcode-designer'
        ) {
            const newDesign = {
                ...this.design,
                [e.detail.name]: e.detail.value,
            }

            console.log(newDesign)

            this.dispatchEvent(
                new CustomEvent('on-input', {
                    bubbles: true,
                    composed: true,
                    detail: {
                        name: 'qrcg-qrcode-designer',
                        value: newDesign,
                    },
                })
            )
        }
        setTimeout(() => {
            this.updatePreview()
        }, 10)
    }

    updatePreview() {
        const doUpdate = () => {
            if (this.urlBuilder.canBuildURL()) {
                this.previewUrl = this.urlBuilder.buildURL()
            } else {
                this.previewUrl = null
            }
        }

        if (this.previewIsRendered) {
            clearTimeout(this.previewUpdateHandle)

            this.previewUpdateHandle = setTimeout(doUpdate, 800)
        } else {
            this.previewIsRendered = true

            doUpdate()
        }
    }

    static validateShapeWith(fnValidator) {
        this.shapeValidator = fnValidator
    }

    shapeIsEnabled(shape) {
        if (this.constructor.shapeValidator)
            return this.constructor.shapeValidator(shape)

        return featureAllowed(`shape.${shape.value}`)
    }

    static validateAdvancedShapeWith(fnValidator) {
        this.advancedShapeValidator = fnValidator
    }

    advancedShapeIsEnabled(shape) {
        if (this.constructor.advancedShapeValidator)
            return this.constructor.advancedShapeValidator(shape)
        return featureAllowed(`advancedShape.${shape.value}`)
    }

    renderSolidTypeInputs() {
        const fillColorInput = html`
            <label>${t`Fill color`}</label>
            <qrcg-color-picker name="foregroundColor"></qrcg-color-picker>
        `

        const eyeColors = html`
            <label>${t`Eye external color`}</label>
            <qrcg-color-picker name="eyeExternalColor"></qrcg-color-picker>

            <label>${t`Eye internal color`}</label>

            <qrcg-color-picker name="eyeInternalColor"></qrcg-color-picker>
        `

        return html`${fillColorInput} ${eyeColors}`
    }

    renderGradientTypeInputs() {
        return html`
            <label>${t`Select gradient type`}</label>
            <qrcg-gradient-input name="gradientFill">
                <span slot="label"> ${t`Fill Gradient`} </span>
            </qrcg-gradient-input>
        `
    }

    renderCurrentTypeInputs() {
        if (!this.design) return

        const renderer = `render${studlyCase(this.design.fillType)}TypeInputs`

        if (isFunction(this[renderer])) return this[renderer]()

        return null
    }

    renderCurrentShapeInputs() {
        if (!this.design.shape) return

        if (this.design.shape === 'none') return

        return html`
            <label>${t`Frame color`}</label>

            <qrcg-color-picker name="frameColor"></qrcg-color-picker>
        `
    }

    renderForegroundImageTypeInputs() {
        return html`
            <label> ${t`Fill image`} </label>

            ${!this.remoteRecord
                ? html` <slot
                      name="file-input-instructions-foreground-image"
                  ></slot>`
                : html`<qrcg-file-input
                      value=${this.remoteRecord?.foreground_image?.id}
                      upload-endpoint="qrcodes/${this.remoteRecord
                          ?.id}/background-image"
                      accept=".png,.jpg,.jpeg"
                      _name="foreground_image"
                  ></qrcg-file-input>`}
        `
    }

    renderSizeWarning() {
        const hasLogo =
            (this.design.logoType === 'preset' && this.design.logoUrl) ||
            (this.design.logoType === 'custom' && this.remoteRecord?.logo)

        const sizeIsLarge =
            (this.design.logoScale > 0.3 && this.design.shape === 'none') ||
            (this.design.logoScale > 0.5 && this.design.shape !== 'none')

        const shouldWarn = hasLogo && sizeIsLarge

        return shouldWarn
            ? html` <div class="warning">
                  ${t`QR Code may not be scannable due to large logo size, test it before you proceed.`}
              </div>`
            : ''
    }

    renderShapeWraning() {
        const connectedModules = 'square,triangle-end,roundness'.split(',')

        const hasDisconnectedModule =
            connectedModules.indexOf(this.design.module) == -1

        const hasCustomFinder =
            this.design.finder != 'default' ||
            this.design.finderDot != 'default'

        const shouldWarn = hasDisconnectedModule && hasCustomFinder

        return shouldWarn
            ? html` <div class="warning">
                  ${t`QR Code may not be scannable due to complex shapes, test it before you proceed.`}
              </div>`
            : ''
    }

    renderCustomEyeShapes() {
        return html`
            <label>${t`Select finder`}</label>

            <qrcg-img-selector
                name="finder"
                value="default"
                .options=${'default,eye-shaped,octagon,rounded-corners,whirlpool,water-drop,circle,zigzag,circle-dots'
                    .split(',')
                    .map((value) => {
                        return {
                            value,
                            src: `${value}.png`,
                        }
                    })}
                base="assets/images/finders"
            ></qrcg-img-selector>

            <label>${t`Select finder dot`}</label>

            <qrcg-img-selector
                name="finderDot"
                value="default"
                .options=${'default,eye-shaped,octagon,rounded-corners,whirlpool,water-drop,circle,zigzag'
                    .split(',')
                    .map((value) => {
                        return {
                            value,
                            src: `${value}.png`,
                        }
                    })}
                base="assets/images/finders/dots"
            ></qrcg-img-selector>
        `
    }

    renderTextInputsForField(field = '') {
        const name = isEmpty(field) ? t`Text` : t(titleCase(field))

        return html`
            <label>${name}</label>
            <qrcg-sticker-text-input
                name=${field}
                .designData=${this.design}
                ?show-text-background-color-input=${this.design.advancedShape !=
                'coupon'}
            ></qrcg-sticker-text-input>
        `
    }

    renderTextInputs() {
        if (this.design.advancedShape === 'coupon') {
            return html`
                ${this.renderTextInputsForField('coupon_text_line_1')}
                ${this.renderTextInputsForField('coupon_text_line_2')}
                ${this.renderTextInputsForField('coupon_text_line_3')}
            `
        }

        return this.renderTextInputsForField('')
    }

    renderAdvancedShapePicker() {
        const picker = html`
            <label>${t`Sticker`}</label>
            <qrcg-img-selector
                name="advancedShape"
                .options=${advancedShapes.map((shape) => ({
                    value: shape.value,
                    src: `${shape.value}.png`,
                    title: shape.name,
                    disabled: !this.advancedShapeIsEnabled(shape),
                }))}
                narrow-padding
                base="assets/images/advanced-shapes"
            ></qrcg-img-selector>
        `

        const fields = html`
            ${this.renderAdvancedShapeFields()}
            <!-- -->
            ${this.renderTextInputs()}
        `

        return html`${picker}
        ${this.design.advancedShape && this.design.advancedShape !== 'none'
            ? fields
            : null}`
    }

    renderAdvancedShapeFields() {
        const functionName = `render${studlyCase(
            this.design.advancedShape
        )}AdvancedShapeFields`

        if (isFunction(this[functionName])) {
            return this[functionName]()
        }
    }

    renderRectFrameTextBottomAdvancedShapeFields() {
        return this.renderRectFrameTextTopAdvancedShapeFields()
    }

    renderRectFrameTextTopAdvancedShapeFields() {
        return html`
            <label>${t`Drop shadow`}</label>

            <qrcg-checkbox name="advancedShapeDropShadow">
                ${this.design.advancedShapeDropShadow
                    ? t`Enabled`
                    : t`Disabled`}
            </qrcg-checkbox>
        `
    }

    renderCouponAdvancedShapeFields() {
        return html`
            <label>${t`Left color`}</label>

            <qrcg-color-picker name="couponLeftColor"></qrcg-color-picker>

            <label>${t`Right color`}</label>

            <qrcg-color-picker name="couponRightColor"></qrcg-color-picker>
        `
    }

    renderFourCornersTextBottomAdvancedShapeFields() {
        return this.renderFourCornersTextTopAdvancedShapeFields()
    }

    renderFourCornersTextTopAdvancedShapeFields() {
        return html`
            <label>${t`Frame color`}</label>

            <qrcg-color-picker
                name="advancedShapeFrameColor"
            ></qrcg-color-picker>
        `
    }

    render() {
        return html`
            <div class="container">
                <div class="row">
                    <div class="input-grid">
                        <label>${t`Select fill type`}</label>
                        <qrcg-balloon-selector
                            name="fillType"
                            value="solid"
                            .options=${[
                                { name: t('Solid color'), value: 'solid' },
                                { name: t('Gradient'), value: 'gradient' },
                                {
                                    name: t('Image'),
                                    value: 'foreground_image',
                                },
                            ]}
                        ></qrcg-balloon-selector>

                        ${this.renderCurrentTypeInputs()}

                        <label>${t`Background`}</label>

                        <qrcg-checkbox name="backgroundEnabled">
                            ${t`Enabled`}
                        </qrcg-checkbox>

                        <label>${t`Background color`}</label>

                        <qrcg-color-picker
                            name="backgroundColor"
                        ></qrcg-color-picker>

                        <label>${t`Select module`}</label>

                        <qrcg-img-selector
                            name="module"
                            value="square"
                            .options=${'square,dots,triangle,rhombus,star-5,star-7,roundness,vertical-lines,horizontal-lines,diamond,fish,tree,twoTrianglesWithCircle,fourTriangles,triangle-end'
                                .split(',')
                                .map((value) => {
                                    return {
                                        value,
                                        src: `${value}.png`,
                                    }
                                })}
                            base="assets/images/modules"
                        ></qrcg-img-selector>

                        ${this.renderCustomEyeShapes()}

                        <label>${t`Select shape`}</label>

                        <qrcg-img-selector
                            name="shape"
                            value="none"
                            .options=${qrShapes.map((shape) => {
                                return {
                                    title: t(shape.name),
                                    value: shape.value,
                                    src: `shape-${shape.value}.jpg`,
                                    disabled: !this.shapeIsEnabled(shape),
                                }
                            })}
                            base="assets/images/shapes"
                        >
                        </qrcg-img-selector>

                        ${this.renderCurrentShapeInputs()}

                        <label>${t`Logo type`}</label>

                        <qrcg-radiogroup name="logoType">
                            <qrcg-radio value="preset">${t`Preset`}</qrcg-radio>
                            <qrcg-radio value="custom"
                                >${t`Your logo`}</qrcg-radio
                            >
                        </qrcg-radiogroup>

                        ${this.design.logoType === 'preset'
                            ? html`
                                  <label>${t`Select logo`}</label>

                                  <qrcg-logo-picker
                                      name="logoUrl"
                                      none-text=${t`NONE`}
                                  ></qrcg-logo-picker>
                              `
                            : html`
                                  <label>${t`Logo`}</label>

                                  ${!this.remoteRecord
                                      ? html` <slot
                                            name="file-input-instructions-logo"
                                        ></slot>`
                                      : html`
                                            <qrcg-file-input
                                                value=${this.remoteRecord?.logo
                                                    ?.id}
                                                upload-endpoint="qrcodes/${this
                                                    .remoteRecord?.id}/logo"
                                                accept=".png,.jpg,.jpeg"
                                                _name="logo"
                                            ></qrcg-file-input>
                                        `}
                              `}
                        ${this.remoteRecord?.logo || this.design.logoUrl
                            ? html`
                                  <label>${t`Logo background`}</label>

                                  <qrcg-checkbox name="logoBackground">
                                      ${this.design.logoBackground
                                          ? t('Enabled')
                                          : t('Disabled')}
                                  </qrcg-checkbox>

                                  <label>${t`Background shape`}</label>

                                  <qrcg-radiogroup name="logoBackgroundShape">
                                      <qrcg-radio value="circle"
                                          >${t`Circle`}</qrcg-radio
                                      >
                                      <qrcg-radio value="square"
                                          >${t`Square`}</qrcg-radio
                                      >
                                  </qrcg-radiogroup>

                                  ${this.design.logoBackground
                                      ? html`
                                            <label
                                                >${t`Background color`}</label
                                            >
                                            <qrcg-color-picker
                                                name="logoBackgroundFill"
                                                value=${this.design
                                                    .logoBackgroundFill}
                                            ></qrcg-color-picker>

                                            <label>${t`Background size`}</label>
                                            <qrcg-range-input
                                                name="logoBackgroundScale"
                                                value=${this.design
                                                    .logoBackgroundScale}
                                                min="0.3"
                                                max="2"
                                                step="0.01"
                                            ></qrcg-range-input>
                                        `
                                      : ''}

                                  <label>${t`Logo size`}</label>

                                  <qrcg-range-input
                                      value=${this.design.logoScale}
                                      name="logoScale"
                                      min=${0}
                                      max=${1}
                                      step=${0.01}
                                  ></qrcg-range-input>

                                  <label>${t`Horizontal position`}</label>

                                  <qrcg-range-input
                                      .value=${this.design.logoPositionX}
                                      name="logoPositionX"
                                      min=${0}
                                      max=${1}
                                      step=${0.01}
                                  ></qrcg-range-input>

                                  <label>${t`Vertical position`}</label>

                                  <qrcg-range-input
                                      .value=${this.design.logoPositionY}
                                      name="logoPositionY"
                                      min=${0}
                                      max=${1}
                                      step=${0.01}
                                  ></qrcg-range-input>

                                  <label>${t`Rotation`}</label>

                                  <qrcg-range-input
                                      .value=${this.design.logoRotate}
                                      name="logoRotate"
                                      min=${0}
                                      max=${360}
                                      step=${1}
                                  ></qrcg-range-input>
                              `
                            : ''}
                        ${this.renderAdvancedShapePicker()}
                    </div>

                    <div class="preview-image">
                        <qrcg-qrcode-image
                            url=${this.previewUrl}
                            ?preview-on-dbl-click=${this.enableLargePreview}
                        ></qrcg-qrcode-image>

                        ${this.renderSizeWarning()}

                        <!-- -->
                        ${this.renderShapeWraning()}

                        <slot name="below-preview-image"></slot>
                    </div>
                </div>
            </div>
        `
    }
}

window.customElements.define('qrcg-qrcode-designer', QRCGQRCodeDesigner)
