import { LitElement, html, css } from 'lit'
import { post } from '../core/api'
import { featureAllowed } from '../core/subscription/logic'
import { t } from '../core/translate'
import { confirm } from '../ui/qrcg-confirmation-modal'
import { showToast } from '../ui/qrcg-toast'

import './qrcg-qrcode-image'
import { openPreviewModal } from './qrcg-qrcode-preview-modal'

class QRCGQRCodeRow extends LitElement {
    static get styles() {
        return css`
            :host {
                display: flex;
                padding: 1rem 1rem 1rem 2rem;
                background-color: var(--gray-0);
                position: relative;
                justify-content: space-between;
                --image-width: 9rem;
                overflow: hidden;
                user-select: none;
                -webkit-user-select: none;
                flex-direction: column;
                transition: opacity 0.3s ease
                    cubic-bezier(0.215, 0.61, 0.355, 1);
            }

            @media (min-width: 900px) {
                :host {
                    flex-direction: row;
                }
            }

            :host([disabled]) {
                opacity: 0.5;
                pointer-events: none;
            }

            qrcg-qrcode-image {
                width: var(--image-width);
                height: var(--image-width);
            }

            h3 {
                margin: 0;
                text-overflow: ellipsis;
                overflow: hidden;
            }

            :host([dynamic]) h3 {
                margin-top: 1rem;
                margin-left: 0.5rem;
            }

            @media (min-width: 900px) {
                h3 {
                    max-width: 300px;
                }
            }

            .name-actions {
                display: flex;
                flex-direction: column;
                justify-content: space-between;
            }

            .dates {
                color: var(--gray-2);
                font-size: 0.8rem;
                /* font-style: italic; */
                line-height: 1.2rem;

                max-width: 10rem;
                text-transform: capitalize;
                margin-top: 1rem;
            }

            @media (min-width: 500px) {
                .dates {
                    max-width: unset;
                }
            }

            .actions {
                display: flex;
                position: relative;
                align-items: center;
                user-select: none;
                -webkit-user-select: none;
                touch-action: manipulation;
                margin-top: 1rem;
            }

            .sep {
                display: block;
                width: 2px;
                background-color: var(--gray-2);
                margin: 0 1rem;
                height: 0.8rem;
            }

            .action {
                color: var(--primary-0);
                text-decoration: underline;
                cursor: pointer;
                display: block;
            }

            :host(:not([dynamic])) .stats {
                opacity: 0.5;
                pointer-events: none;
                text-decoration: none;
            }

            .type {
                font-size: 0.7rem;
                background-color: var(--gray-2);
                color: white;
                padding: 0.5rem 1rem;
                border-radius: 0.5rem;
                text-transform: uppercase;
                display: inline-flex;
                margin-top: 1rem;
            }

            .dynamic-badge {
                position: absolute;
                font-size: 0.7rem;
                top: 0;
                left: 0;
                background-color: var(--primary-0);
                color: white;
                padding: 0.25rem 1.5rem;
                transform: translate(-25px, 16px) rotate(-45deg);
                text-transform: uppercase;
            }

            qrcg-qrcode-image {
                cursor: pointer;
                margin: 2rem auto 1rem;
                position: relative;
                left: -1rem;
            }

            @media (min-width: 900px) {
                qrcg-qrcode-image {
                    margin: 0;
                    left: 0;
                }
            }
        `
    }

    static get properties() {
        return {
            qrcode: { type: Object },
            isDynamic: { type: Boolean, reflect: true, attribute: 'dynamic' },
        }
    }

    connectedCallback() {
        super.connectedCallback()
        this.isDynamic = this.qrcode.isDynamic()
    }

    willUpdate(changed) {
        if (changed.has('qrcode')) {
            this.isDynamic = this.qrcode.isDynamic()
        }
    }

    toggleArchive() {
        const shouldToggle = this.dispatchEvent(
            new CustomEvent('qrcg-qrcode-row:will-toggle-archive', {
                detail: { qrcode: this.qrcode },
                cancelable: true,
                composed: true,
                bubbles: true,
            })
        )

        if (!shouldToggle) return

        this.dispatchEvent(
            new CustomEvent('on-toggle-archive', {
                detail: { qrcode: this.qrcode },
            })
        )
    }

    async onCopy() {
        if (!featureAllowed('qrcode.copy')) {
            return await confirm({
                title: t`Upgrade Required`,
                message: t`This feature is not available on your current plan.`,
                affirmativeText: t`OK`,
                negativeText: null,
            })
        }

        await confirm({
            message: t`Are you sure you want to copy (${this.qrcode.name})?`,
        })

        const message = [
            t`Copying QR code`,
            ' ',
            t`in a background process ...`,
        ].join('')

        this.fire('qrcg-qrcode-row:before-copy')

        const toastPromise = showToast(message, 2000)

        await post(`qrcodes/${this.qrcode.id}/copy`)

        await toastPromise

        await new Promise((resolve) => setTimeout(resolve, 100))

        showToast(t`QR code copied successfully ...`)

        this.fire('qrcg-qrcode-row:after-copy')
    }

    openPreviewModal() {
        try {
            openPreviewModal({
                qrcode: this.qrcode,
            })
        } catch {
            //
        }
    }

    fire(eventName, detail = {}) {
        this.dispatchEvent(
            new CustomEvent(eventName, {
                composed: true,
                bubbles: true,
                detail: {
                    ...detail,
                    qrcode: this.qrcode,
                },
            })
        )
    }

    render() {
        return html`
            ${this.qrcode.isDynamic()
                ? html`<div class="dynamic-badge">${t`Dynamic`}</div>`
                : ''}

            <div class="name-actions">
                <h3>${this.qrcode.getName()}</h3>
                <div class="dates">${this.qrcode.getDates()}</div>
                <div class="type-container">
                    <div class="type">${t(this.qrcode.type)}</div>
                </div>
                <div class="actions">
                    <a class="action" href=${this.qrcode.getEditLink()}
                        >${t`Edit`}</a
                    >
                    <span class="sep"></span>

                    <a class="action stats" href=${this.qrcode.getStatsLink()}
                        >${t`Stats`}</a
                    >
                    <span class="sep"></span>

                    <a class="action" @click=${this.toggleArchive}>
                        ${this.qrcode.archived ? t('Restore') : t('Archive')}
                    </a>
                    <span class="sep"></span>

                    <a class="action" @click=${this.onCopy}> ${t`Copy`} </a>
                </div>
            </div>
            <qrcg-qrcode-image
                @click=${this.openPreviewModal}
                url=${this.qrcode.getFileUrl('svg')}
                convert-to-png
            ></qrcg-qrcode-image>
        `
    }
}

window.customElements.define('qrcg-qrcode-row', QRCGQRCodeRow)
