import { css, html, LitElement } from 'lit'

import { get } from '../core/api'

import { hexToRgb, isEmpty } from '../core/helpers'

import '../ui/qrcg-chart'

import '../ui/qrcg-loader'

import { QRCGRouteParamsController } from '../core/qrcg-route-params-controller'
import { t } from '../core/translate'

const colors = [
    '034732',
    '008148',
    'c6c013',
    'ef8a17',
    'ef2917',
    '483c46',
    '3c6e71',
    '70ae6e',
    'beee62',
    'f4743b',
]
    .map(hexToRgb)
    .map((c) => {
        return `rgba( ${c.r}, ${c.g}, ${c.b}`
    })
    .map((c) => {
        return {
            backgroundColor: `${c}, 0.3)`,
            borderColor: `${c}, 1)`,
        }
    })

export class QrcgQrcodeStats extends LitElement {
    routeParams = new QRCGRouteParamsController(this)

    static get styles() {
        return css`
            :host {
                display: block;
            }

            qrcg-chart {
                max-width: 800px;
                margin: auto;
            }

            qrcg-loader {
                margin: auto;
                display: block;
                width: fit-content;
            }
        `
    }

    static get properties() {
        return {
            id: {},
            chartData: { type: Array },
            loading: { type: Boolean },
        }
    }

    firstUpdated() {
        this.id = this.routeParams.get('id')
    }

    onRouteParamChange(params) {
        this.id = params.id
    }

    updated(changed) {
        if (changed.has('id') && this.id) {
            this.fetchResults()
        }
    }

    async fetchResults() {
        this.loading = true

        const { response } = await get(`qrcodes/stats/${this.id}`)

        const json = await response.json()

        this.chartData = json.map((row, key) => {
            return {
                label: row.date,
                number: row.scans,
                ...colors[key % colors.length],
            }
        })

        this.loading = false
    }

    render() {
        return html`${this.loading
            ? html`<qrcg-loader></qrcg-loader>`
            : isEmpty(this.chartData)
            ? html`<p>${t`This QR Code did not get any scans yet.`}</p>`
            : html`<qrcg-chart .data=${this.chartData}></qrcg-chart>`}`
    }
}

customElements.define('qrcg-qrcode-stats', QrcgQrcodeStats)
