import { LitElement, css, html } from 'lit'

import { t } from '../../core/translate'

import { QRCGApiConsumer } from '../../core/qrcg-api-consumer'

import { QrcgTheme } from '../../ui/qrcg-theme'

import { showToast } from '../../ui/qrcg-toast'

export class QrcgSystemSettingsFormBase extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
            }

            [name] {
                margin-top: 1rem;
            }

            section {
                border: 0.5rem solid var(--gray-0);
                padding: 1rem;
                margin-bottom: 1rem;
            }

            .section-title {
                margin-top: 0;
                /* font-weight: normal; */
                color: var(--gray-2);
            }

            qrcg-form-comment p {
                margin: 0;
            }

            @media (min-width: 900px) {
                qrcg-input {
                    max-width: 50%;
                }
            }

            .actions {
                display: flex;
                justify-content: flex-end;
            }

            a {
                color: var(--primary-0);
                text-decoration: none;
            }
        `,
    ]

    static get properties() {
        return {
            configs: { type: Array },
        }
    }

    constructor() {
        super()

        this.api = QRCGApiConsumer.instance({
            host: this,
            baseRoute: 'system/configs',
            bindEvents: true,
            loadableElementsSelector: 'qrcg-button',
            disableableInputsSelector: '[name]',
        })
    }

    connectedCallback() {
        super.connectedCallback()

        this.addEventListener('on-input', this.onInput)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('on-input', this.onInput)
    }

    firstUpdated() {
        this.fetchConfigs()
    }

    async fetchConfigs() {
        this.configs = await this.api.search({
            keys: this.inputs.map((elem) => elem.name),
        })
    }

    updated(changed) {
        if (changed.has('configs')) {
            this.configs.forEach((item) => {
                const elem = this.shadowRoot.querySelector(
                    `[name="${item.key}"]`
                )

                if (elem) elem.value = item.value
            })
        }
    }

    onApiSuccess(e) {
        const request = e.detail.request

        if (request.method === 'POST') {
            showToast(t`Settings saved successfully.`)
        }
    }

    get inputs() {
        return Array.from(this.shadowRoot.querySelectorAll('[name]'))
    }

    onInput(e) {
        const elem = e.composedPath()[0]

        if (elem.name?.match(/theme/)) {
            QrcgTheme.setThemeConfig({
                key: elem.name,
                value: elem.value,
            })
        }

        this.setLocalConfig(e.detail.name, e.detail.value)
    }

    setLocalConfig(key, value) {
        this.configs = this.configs.map((item) => {
            if (item.key === key) {
                return {
                    key,
                    value,
                }
            }
            return item
        })
    }

    async saveConfigs() {
        const data = this.inputs.map((elem) => ({
            key: elem.name,
            value: elem.value,
        }))

        await this.api.post('system/configs', data)
    }

    renderForm() {}

    renderFileInput({ label, name, instructions, accept }) {
        const s = name.split('.')

        if (!accept) accept = `.${s[s.length - 1]}`

        return html`
            <qrcg-file-input
                name="${name}"
                upload-endpoint="system/configs/upload?key=${name}"
                accept=${accept}
            >
                ${t(label)}
                ${instructions
                    ? html` <div slot="instructions">${t(instructions)}</div>`
                    : null}
            </qrcg-file-input>
        `
    }

    renderActions() {
        return html`
            <div class="actions">
                <qrcg-button class="save-button" @click=${this.saveConfigs}>
                    ${t`Save`}
                </qrcg-button>
            </div>
        `
    }

    render() {
        return html`
            ${this.renderForm()}
            <!-- -->
            ${this.renderActions()}
        `
    }
}
