import { LitElement, css } from 'lit'
import { unsafeStatic, html } from 'lit/static-html.js'

import { Config } from '../core/qrcg-config'

let sloganLetterSpacing

export class QrcgAppLogo extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
                position: relative;
                color: white;
            }

            h2 {
                letter-spacing: 3px;
                font-weight: bold;
                margin: 0;
                font-size: 1.3rem;
                text-transform: uppercase;

                display: flex;
            }

            .slogan {
                display: flex;
                font-size: 0.6rem;
                margin-top: 0.2rem;
                text-transform: uppercase;
                padding-left: 0.1rem;
                opacity: 0;
            }
            a {
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
            }

            .logo-img {
                max-height: var(--dashboard-header-height);
                max-width: 12rem;
            }
        `,
    ]

    static get properties() {
        return {
            variation: {},
            sloganLetterSpacing: {},
            href: {
                reflect: true,
            },
        }
    }

    connectedCallback() {
        super.connectedCallback()
    }

    firstUpdated() {
        if (!this.logo) this.calculateSloganLetterSpacing()
    }

    async calculateSloganLetterSpacing() {
        await new Promise((resolve) => setTimeout(resolve, 0))

        const name = this.renderRoot.querySelector('h2')

        const nameWidth = name.clientWidth

        if (nameWidth === 0) {
            return
        }

        const slogan = this.renderRoot.querySelector('.slogan')

        const sloganClone = slogan.cloneNode(true)

        sloganClone.style =
            'position: absolute; top: 0; left: 0; opacity: 0; z-index: 999999; color: black; border: 1px solid black; width: max-content;'

        sloganClone.id = 'slogan-clone'

        this.shadowRoot.appendChild(sloganClone)

        const step = 0.001

        const accuracy = 1.5

        let i = 0,
            difference,
            counter = 0

        const measure = async () => {
            if (difference < accuracy) {
                sloganLetterSpacing = `${i}rem`

                sloganClone.remove()

                this.requestUpdate()
                return
            }

            if (++counter > 1000) {
                console.error('maximum iteration reached, breaking')
                return
            }

            i += step

            sloganClone.style.letterSpacing = `${i}rem`

            difference = nameWidth - sloganClone.clientWidth

            measure()
        }

        measure()
    }

    renderLogoImage() {
        return html`
            ${this.renderLink()}

            <img class="logo-img" src=${this.logo} part="logo-img" />
        `
    }

    get logo() {
        const regular = Config.get('frontend.header_logo_url')

        const inverse = Config.get('frontend.header_logo_inverse_url')

        return this.variation === 'inverse' ? inverse : regular
    }

    renderLink() {
        const config = Config.get('app.frontend_links')

        const frontendLinksDisabled = config === 'disabled'

        if (frontendLinksDisabled) {
            return
        }

        return html`${this.href ? html`<a href="${this.href}"></a>` : ''}`
    }

    render() {
        if (this.logo) {
            return this.renderLogoImage()
        }
        let nameStyle = '',
            sloganStyle = ''

        if (this.variation === 'inverse') {
            nameStyle = unsafeStatic`style="color: black"`
            sloganStyle = 'color: black'
        }

        if (sloganLetterSpacing) {
            sloganStyle = unsafeStatic(
                `style="${sloganStyle}; letter-spacing: ${sloganLetterSpacing}; opacity: 1"`
            )
        } else {
            sloganStyle = unsafeStatic(`style=${sloganStyle};`)
        }

        const appName = Config.get('app.name')

        const slogan = Config.get('frontend.slogan')

        return html`
            ${this.renderLink()}

            <h2 ${nameStyle}>${appName}</h2>

            <span ${sloganStyle} class="slogan"> ${slogan} </span>
        `
    }
}
customElements.define('qrcg-app-logo', QrcgAppLogo)
