import { LitElement, html, css } from 'lit'
import { isEmpty } from '../core/helpers'

import './qrcg-loader-h'

export class QRCGButton extends LitElement {
    static get styles() {
        return css`
            :host {
                display: flex;
                position: relative;
                transition: opacity 0.5s ease-in-out;
                cursor: pointer;
                -webkit-tap-highlight-color: transparent;
                --button-color: white;
            }

            :host([hidden]) {
                display: none;
            }

            :host([disabled]),
            :host([loading]) {
                pointer-events: none;
                opacity: 0.8;
            }

            button {
                padding: 0.8rem 1rem;
                width: 100%;
                font-weight: bold;
                background-color: var(--primary-0);
                color: var(--button-color);
                display: flex;
                align-items: center;
                justify-content: center;
                -webkit-appearance: none;
                border: 0;
                cursor: pointer;
                font-weight: bold;
                font-size: 0.8rem;
                letter-spacing: 1px;
                min-width: 110px;
                user-select: none;
                -webkit-user-select: none;
                -webkit-tap-highlight-color: transparent;
                /** prevent zoom on multiple tap */
                touch-action: manipulation;
                border-radius: 0.5rem;
                outline: 0;
                position: relative;
            }

            button:focus {
                color: var(--button-color);
                outline: 2px solid var(--primary-1);
            }

            button:hover {
                background-color: var(--primary-1);
            }

            button:active {
                color: var(--button-color);
            }

            :host([transparent]) button {
                background-color: transparent;
                color: black;
            }

            :host([transparent]) button:hover {
                background-color: var(--gray-1);
                color: black;
            }

            qrcg-loader-h {
                position: absolute;
                pointer-events: none;
                transform: translateY(-50%) scale(0.8);
                top: calc(50% + 2px);
                display: none;
            }

            :host([loading]) qrcg-loader-h {
                display: block;
            }

            :host([loading]) .content {
                opacity: 0;
            }

            a {
                position: absolute;
                top: 0;
                left: 0;
                bottom: 0;
                right: 0;
            }
        `
    }

    static get properties() {
        return {
            type: {},
            loading: {
                type: Boolean,
                reflect: true,
            },
            disabled: {
                type: Boolean,
            },
            href: {},
            target: {},
            transparent: {
                type: Boolean,
                reflect: true,
            },
            download: { reflect: true },
        }
    }

    connectedCallback() {
        super.connectedCallback()
    }

    updated(changed) {
        if (changed.has('href') || changed.has('download')) {
            this.updateLink()
        }
    }

    updateLink() {
        const link = this.renderRoot.querySelector('a')

        if (!link) return

        if (isEmpty(this.download)) {
            link.removeAttribute('download')
        } else {
            link.setAttribute('download', this.download)
        }
    }

    render() {
        return html`
            <button
                type=${this.type}
                part="button"
                .disabled=${this.disabled || this.loading}
            >
                ${this.href
                    ? html`<a
                          href=${this.href}
                          target="${this.target}"
                          .download=${this.download}
                          tabindex="-1"
                      ></a>`
                    : ''}
                <qrcg-loader-h></qrcg-loader-h>
                <div class="content" part="content">
                    <slot></slot>
                </div>
            </button>
        `
    }
}

window.customElements.define('qrcg-button', QRCGButton)
