import { LitElement, html, css } from 'lit'
import { styled } from '../core/helpers'
import { t } from '../core/translate'

import './qrcg-box'

export class QrcgFaqs extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
                line-height: 1.8;
                user-select: none;
                -webkit-user-select: none;
                padding: 1rem;
            }

            .container {
                max-width: var(--container-max-width);
                margin: auto;
            }

            header {
                text-align: center;
            }

            .description {
                color: var(--gray-2);
            }
        `,
    ]

    static get properties() {
        return {}
    }

    constructor() {
        super()

        this.styleId = this.tagName + '-styles'

        this.onClick = this.onClick.bind(this)

        this.parentMatches = this.parentMatches.bind(this)
    }

    connectedCallback() {
        super.connectedCallback()
        this.attachStyles()
        this.addEventListener('click', this.onClick)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.detachStyles()
        this.removeEventListener('click', this.onClick)
    }

    setMaxHeight(maxHeight) {
        let style = document.head.querySelector(`#${this.styleId}`)

        style.innerHTML = styled`
            qrcg-faqs qrcg-faq-title.active + qrcg-faq-content {
                max-height: ${maxHeight}px;
            }
        `
    }

    attachStyles() {
        const elem = document.createElement('style')

        elem.id = this.styleId

        document.head.appendChild(elem)

        return elem
    }

    detachStyles() {
        const ids = [this.styleId]

        ids.forEach((id) => document.head.querySelector(`#${id}`).remove())
    }

    onClick(e) {
        const title = this.parentMatches(e.target, 'qrcg-faq-title')

        if (title) this.handleTitleClick(title)
    }

    async handleTitleClick(faqTitle) {
        const isActive = faqTitle.matches('.active')

        this.deactivateAllfaqTitles()

        if (!isActive) {
            const faqContent = faqTitle.nextElementSibling

            const height = await this.getContentHeight(faqContent)

            this.setMaxHeight(height)

            await new Promise((resolve) => setTimeout(resolve, 0))

            faqTitle.classList.add('active')
        }
    }

    async getContentHeight(faqContent) {
        const clone = faqContent.cloneNode(true)

        clone.style = styled`
            opacity: 0;
            max-height: initial;
            pointer-events: none;
            position: fixed;
            width: ${faqContent.clientWidth}px;
            z-index: 10000;
            top: 0;
            left: 0;
        `

        this.appendChild(clone)

        await new Promise((resolve) => setTimeout(resolve, 0))

        const height = clone.clientHeight

        clone.remove()

        return height
    }

    deactivateAllfaqTitles() {
        this.querySelectorAll('qrcg-faq-title').forEach((elem) => {
            elem.classList.remove('active')
        })
    }

    parentMatches(elem, selector) {
        if (elem.matches(selector)) {
            return elem
        }

        let parent = elem

        while (parent.tagName !== this.tagName) {
            if (parent.matches(selector)) {
                return parent
            }

            parent = elem.parentElement
        }

        return false
    }

    render() {
        return html`
            <div class="container">
                <header>
                    <qrcg-h2>
                        <slot name="title"> ${t`FAQs`} </slot>
                    </qrcg-h2>
                    <qrcg-p class="description">
                        <slot name="description">
                            ${t`Basic questions and answers`}
                        </slot>
                    </qrcg-p>
                </header>
                <qrcg-box>
                    <slot name="content"></slot>
                </qrcg-box>
            </div>
        `
    }
}
customElements.define('qrcg-faqs', QrcgFaqs)
