import { css, html } from 'lit'
import { BaseInput } from './base-input'

class QRCGInput extends BaseInput {
    static get styles() {
        return [
            super.styles,
            css`
                :host([prefix]) input {
                    padding-left: 2.8rem;
                }

                .prefix {
                    position: absolute;
                    background-color: var(--gray-1);
                    color: var(--gray-2);
                    left: 0rem;
                    border-top-left-radius: 0.5rem;
                    border-bottom-left-radius: 0.5rem;
                    top: 0.5rem;
                    bottom: 0.5rem;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-size: 0.8rem;
                    padding: 0 0.8rem;
                }

                :host([focused]) .prefix {
                    background-color: var(--gray-2);
                    color: white;
                }
            `,
        ]
    }

    static get properties() {
        return {
            ...super.properties,
            prefix: {
                reflect: true,
            },
            focused: {
                reflect: true,
                type: Boolean,
            },
            min: {},
            max: {},
            step: {},
        }
    }

    renderPrefix() {
        if (!this.input) return

        if (this.prefix)
            return html`
                <span class="prefix" style=${`top: ${this.input?.offsetTop}px`}>
                    ${this.prefix}
                </span>
            `
    }

    firstUpdated() {
        super.firstUpdated()

        this.bindFocus()
        this.requestUpdate()
    }

    updated(changed) {
        this.bindInputNodeAttributes(changed)
    }

    bindInputNodeAttributes(changed) {
        if (!this.input) return

        const boundAttributes = ['min', 'max', 'step']

        boundAttributes.forEach((prop) => {
            if (changed.has(prop)) {
                this.input.setAttribute(prop, this[prop])
            }
        })
    }

    bindFocus() {
        this.input.addEventListener('focus', this.onFocus)
        this.input.addEventListener('blur', this.onBlur)
    }

    detachFocus() {
        this.input.removeEventListener('focus', this.onFocus)
        this.input.removeEventListener('blur', this.onBlur)
    }

    onBlur = () => {
        this.focused = false
    }

    onFocus = () => {
        this.focused = true
    }

    get input() {
        return this.shadowRoot.querySelector('input')
    }

    disconnectedCallback() {
        super.disconnectedCallback()

        this.detachFocus()
    }

    render() {
        return html` ${super.render()} ${this.renderPrefix()} `
    }
}

window.customElements.define('qrcg-input', QRCGInput)
