import { LitElement, html, css } from 'lit'
import { debounce, isEmpty } from '../core/helpers'

export class QrcgRangeInput extends LitElement {
    static styles = [
        css`
            :host {
                display: flex;
                position: relative;
            }

            input {
                display: block;
                flex: 1;
            }

            .value {
                display: flex;
            }

            .number {
                width: 2rem;
            }

            /** Compiled CSS from https://codepen.io/davidkpiano/pen/xLKBpM */

            [type='range'] {
                -webkit-appearance: none;
                margin: 0.75rem 0;
                width: 100%;
            }
            [type='range']:focus {
                outline: 0;
            }
            [type='range']:focus::-webkit-slider-runnable-track {
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
            }
            [type='range']:focus::-ms-fill-lower {
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
            }
            [type='range']:focus::-ms-fill-upper {
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
            }
            [type='range']::-webkit-slider-runnable-track {
                cursor: pointer;
                height: 0.4rem;
                -webkit-transition: all 0.2s ease;
                transition: all 0.2s ease;
                width: 100%;
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
                border: 0.1rem solid #dedede;
                border-radius: 25px;
            }
            [type='range']::-webkit-slider-thumb {
                background: var(--primary-0);
                border-radius: 0.2rem;
                cursor: pointer;
                height: 1.5rem;
                width: 1.5rem;
                -webkit-appearance: none;
                margin-top: -0.65rem;
            }
            [type='range']::-moz-range-track {
                cursor: pointer;
                height: 0.4rem;
                -moz-transition: all 0.2s ease;
                transition: all 0.2s ease;
                width: 100%;
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
                border: 0.1rem solid #dedede;
                border-radius: 25px;
            }
            [type='range']::-moz-range-thumb {
                background: var(--primary-0);
                border-radius: 0.2rem;
                cursor: pointer;
                height: 1.5rem;
                width: 1.5rem;
            }
            [type='range']::-ms-track {
                cursor: pointer;
                height: 0.4rem;
                -ms-transition: all 0.2s ease;
                transition: all 0.2s ease;
                width: 100%;
                background: transparent;
                border-color: transparent;
                border-width: 0.75rem 0;
                color: transparent;
            }
            [type='range']::-ms-fill-lower {
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
                border: 0.1rem solid #dedede;
                border-radius: 50px;
            }
            [type='range']::-ms-fill-upper {
                background: linear-gradient(
                    to bottom,
                    #dedede,
                    #dedede 49.9%,
                    #d1d1d1 50%,
                    #d1d1d1 100%
                );
                border: 0.1rem solid #dedede;
                border-radius: 50px;
            }
            [type='range']::-ms-thumb {
                background: var(--primary-0);
                border-radius: 0.2rem;
                cursor: pointer;
                height: 1.5rem;
                width: 1.5rem;
                margin-top: 0;
            }
        `,
    ]

    static get properties() {
        return {
            name: {},
            value: {},
            min: {},
            max: {},
            step: {},
            showValue: {
                attribute: 'show-value',
                type: Boolean,
            },
        }
    }

    constructor() {
        super()

        this.onInput = debounce(this.onInput, 500)
    }

    connectedCallback() {
        super.connectedCallback()
        this.addEventListener('input', this.onInput)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('input', this.onInput)
    }

    updated(change) {
        if (change.has('value')) this.rangeInput.value = this.getValue()
    }

    get rangeInput() {
        return this.shadowRoot.querySelector('input')
    }

    onInput = () => {
        const value = this.rangeInput.value

        this.dispatchEvent(
            new CustomEvent('on-input', {
                composed: true,
                bubbles: true,
                detail: {
                    name: this.name,
                    value,
                },
            })
        )
    }

    getValue() {
        if (isEmpty(this.value)) {
            return '0'
        }

        return `${this.value}`
    }

    render() {
        return html`
            <input
                type="range"
                min=${this.min}
                max=${this.max}
                step=${this.step}
            />

            ${this.showValue
                ? html`
                      <div class="value">
                          <div class="number">${this.value}</div>
                      </div>
                  `
                : ''}
        `
    }
}

customElements.define('qrcg-range-input', QrcgRangeInput)
