import { LitElement, html, css } from 'lit'
import { isFunction } from '../core/helpers'

import './qrcg-box'

export class QrcgToast extends LitElement {
    static styles = [
        css`
            :host {
                display: block;
                position: fixed;
                width: max-content;
                bottom: 2rem;
                left: 50%;
                transform: translateX(-50%);
                animation: fadeIn 0.3s ease-in both;
                line-height: 1.5;
            }

            :host([hidden]) {
                animation: fadeOut 0.3s ease-out both;
            }

            qrcg-box {
                background-color: black;
                color: white;
            }

            @keyframes fadeIn {
                from {
                    opacity: 0;
                }

                to {
                    opacity: 1;
                }
            }

            @keyframes fadeOut {
                from {
                    opacity: 1;
                }

                to {
                    opacity: 0;
                }
            }
        `,
    ]

    constructor() {
        super()

        this.removeSelf = this.removeSelf.bind(this)
        this.onAnimationEnd = this.onAnimationEnd.bind(this)
    }

    static get properties() {
        return {
            time: { type: Number },
            hidden: { type: Boolean, reflect: true },
        }
    }

    connectedCallback() {
        super.connectedCallback()

        setTimeout(this.removeSelf, this.time)

        this.addEventListener('animationend', this.onAnimationEnd)
    }

    disconnectedCallback() {
        super.disconnectedCallback()
        this.removeEventListener('animationend', this.onAnimationEnd)
    }

    removeSelf() {
        this.hidden = true
    }

    onAnimationEnd() {
        if (this.hidden) {
            this.remove()

            if (isFunction(this.onAfterRemove)) {
                this.onAfterRemove()
            }
        }
    }

    render() {
        return html`
            <qrcg-box>
                <slot></slot>
            </qrcg-box>
        `
    }
}

export function showToast(message, time = 5000) {
    return new Promise((resolve) => {
        const toast = new QrcgToast()

        toast.onAfterRemove = resolve

        toast.innerHTML = message

        toast.time = time

        document.body.appendChild(toast)
    })
}

customElements.define('qrcg-toast', QrcgToast)
