<?php

namespace App\Http\Controllers;

use App\Models\Config;
use App\Models\User;

use Illuminate\Http\Request;

use Illuminate\Validation\ValidationException;

use Illuminate\Validation\Rules\Password as PasswordRule;

use Illuminate\Support\Facades\Hash;

use Illuminate\Support\Facades\Password;

use Illuminate\Auth\Events\Registered;

use Illuminate\Auth\Events\Verified;

use Illuminate\Auth\Notifications\ResetPassword;

use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Facades\Validator;

class AccountController extends Controller
{
    public function register(Request $request)
    {
        $registrationDisabled = Config::get('app.new_user_registration') == 'disabled';

        $validator = Validator::make($request->all(), $registrationDisabled ? [] : [
            'name' => 'required',
            'email' => 'required|email|unique:App\Models\User',
            'password' => ['required', 'confirmed', PasswordRule::min(6)]
        ]);

        $validator->after(function ($validator) use ($registrationDisabled) {
            if ($registrationDisabled) {
                $validator->errors()->add('email', t('New registrations are disabled.'));
            }
        });

        $validator->validate();

        $user = User::create(array_merge($request->all(), [
            'password' => Hash::make($request->password)
        ]));

        if (!config('app.email_verification_after_sign_up')) {
            $user->email_verified_at = now();
            $user->save();
        }

        // Sends the verification email
        event(new Registered($user));

        return [
            'token' => $user->createToken('spa')->plainTextToken
        ];
    }

    public function resendVerificationEmail(Request $request)
    {
        $request->user()->sendEmailVerificationNotification();

        return [
            'sent' => true
        ];
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {

            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        if (!$user->email_verified_at) {
            throw ValidationException::withMessages([
                'email' => ['You must verify your email before login']
            ]);
        }

        $user = User::with(
            'subscriptions',
            'subscriptions.subscription_plan',
            'subscriptions.statuses'
        )->find($user->id);

        return [
            'token' => $user->createToken('spa')->plainTextToken,
            'user' => $user
        ];
    }

    public function verifyEmail(Request $request, $id, $hash)
    {
        $user = User::find($id);

        if (!$user) {
            abort(403, 'Invalid user');
        }

        if (!hash_equals(
            (string) $request->route('hash'),
            sha1($user->getEmailForVerification())
        )) {

            abort(403, 'Invalid hash code');
        }


        if (!$user->hasVerifiedEmail()) {

            $user->markEmailAsVerified();

            event(new Verified($user));
        }

        return redirect(
            config('frontend.url') . '/account/email-verified'
        );
    }

    public function forgotPassword(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        ResetPassword::createUrlUsing(function ($notifiable, $token) {
            $email = $notifiable->getEmailForPasswordReset();

            return config('frontend.url')
                . "/account/reset-password?token=$token&email=$email";
        });

        Password::sendResetLink(
            $request->only('email')
        );

        return [
            'success' => 1
        ];
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:6|confirmed',
        ]);


        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {

                $user->forceFill([
                    'password' => Hash::make($password)
                ]);

                $user->save();

                event(new PasswordReset($user));
            }
        );

        return compact('status');
    }

    public function myself(Request $request)
    {
        $user = User::with(
            'subscriptions',
            'subscriptions.subscription_plan',
            'subscriptions.statuses'
        )->find(
            $request->user()->id
        );

        return $user;
    }
}
