<?php

namespace App\Http\Controllers;

use App\Events\QRCodeLogoUploaded;
use App\Events\ShouldSaveQRCodeVariants;
use App\Http\Requests\QRCodeRequest;

use App\Http\Requests\ArchiveQRCodeRequest;
use App\Interfaces\FileManager;
use App\Models\QRCode;

use Illuminate\Http\Request;

use App\Interfaces\QRCodeGenerator;

use App\Interfaces\QRCodeStats;
use App\Support\QRCodeManager;
use Illuminate\Support\Facades\DB;


class QRCodeController extends Controller
{
    private QRCodeGenerator $generator;

    public function __construct(QRCodeGenerator $generator)
    {
        $this->generator = $generator;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = QRCode::query();

        $query->whereArchived($request->boolean('search_archived'));

        if (!empty($request->search_name))

            $query->where(
                DB::raw('lower(name)'),
                'like',
                '%' . strtolower($request->search_name) . '%'
            );

        if (!empty($request->search_type))

            $query->where('type', $request->search_type);

        if ($request->user()->cannot('listAll', QRCode::class)) {
            $query->where('user_id', $request->user()->id);
        }

        $query->orderBy('id', 'desc');

        $paginated = $query->paginate(10);

        call_user_func([$paginated, 'withPath'], $request->path);

        return $paginated;
    }

    public function show(QRCode $qrcode)
    {
        return $qrcode;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\QRCodeRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(QRCodeRequest $request)
    {
        $qrcode = new QRCode($request->all());

        $qrcode->user_id = $request->user()->id;

        $qrcode->save();

        return $qrcode;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\QRCodeRequest  $request
     * @param  \App\Models\QRCode  $qrcode
     * @return \Illuminate\Http\Response
     */
    public function update(QRCodeRequest $request, QRCode $qrcode)
    {
        $qrcode->fill($request->all())->save();

        return $qrcode;
    }

    public function archive(ArchiveQRCodeRequest $request, QRCode $qrcode)
    {
        $qrcode->archived = $request->boolean('archived');

        $qrcode->save();

        return $qrcode;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\QRCode  $qrcode
     * @return \Illuminate\Http\Response
     */
    public function destroy(QRCode $qrcode)
    {
        // Deleting QR Codes is not supported for now.
        // Archive instead.
    }

    public function storeLogo(Request $request, QRCode $qrcode, FileManager $files)
    {
        $request->merge([
            'attachable_type' => QRCode::class,
            'attachable_id' => $qrcode->id,
            'type' => FileManager::FILE_TYPE_QRCODE_LOGO
        ]);

        $result = $files->store($request);

        event(new QRCodeLogoUploaded($qrcode));

        $qrcode->touch();

        return $result;
    }

    public function storeForegroundImage(Request $request, QRCode $qrcode, FileManager $files)
    {
        $request->merge([
            'attachable_type' => QRCode::class,
            'attachable_id' => $qrcode->id,
            'type' => FileManager::FILE_TYPE_QRCODE_FOREGROUND_IMAGE
        ]);

        $result = $files->store($request);

        event(new ShouldSaveQRCodeVariants($qrcode));

        $qrcode->touch();

        return $result;
    }

    public function preview(Request $request)
    {
        $this->generator->initFromRequest($request);

        return $this->generator->respondInline();
    }

    public function stats(Request $request, QRCodeStats $stats, QRCode $qrcode)
    {
        return $stats->getStats($qrcode);
    }

    public function copy(QRCode $qrcode)
    {
        $qrcodes = new QRCodeManager();

        return $qrcodes->copy($qrcode);
    }
}
