<?php

namespace App\Http\Controllers;

use App\Events\Subscribed;
use App\Events\SubscriptionVerified;
use App\Http\Requests\StoreSubscriptionRequest;
use App\Http\Requests\UpdateSubscriptionRequest;
use App\Http\Resources\SubscriptionResource;
use App\Interfaces\CurrencyManager;
use App\Interfaces\ModelIndex;
use App\Interfaces\SubscriptionManager;
use App\Models\Subscription;
use App\Models\SubscriptionStatus;
use App\Repositories\PaypalPaymentGateway;
use Illuminate\Http\Request;
use InvalidArgumentException;

class SubscriptionController extends Controller
{
    private SubscriptionManager $subscriptions;

    public function __construct(SubscriptionManager $subscriptions)
    {
        $this->subscriptions = $subscriptions;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(ModelIndex $index, Request $request)
    {
        return $index
            ->build(new Subscription(), $request)
            ->withQuery(function ($query) {
                $query->with(
                    'user',
                    'subscription_plan',
                    'statuses'
                );
            })
            ->list(
                [
                    'user.name',
                    'subscription_plan.name',
                    'statuses.status'
                ]
            );
    }

    public function subscribe(Request $request)
    {
        $subscription = Subscription::where('user_id', $request->user()->id)
            ->where('subscription_plan_id', $request->subscription_plan_id)
            ->first();

        if (!$subscription) {
            $subscription = new Subscription(
                [
                    'user_id' => $request->user()->id,
                    'subscription_plan_id' => $request->subscription_plan_id
                ]
            );

            $subscription->save();

            event(new Subscribed($subscription));
        }

        return $subscription;
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreSubscriptionRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreSubscriptionRequest $request)
    {
        return new SubscriptionResource(
            $this->subscriptions->saveSubscription($request->all())
        );
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Subscription  $subscription
     * @return \Illuminate\Http\Response
     */
    public function show(Subscription $subscription)
    {
        return new SubscriptionResource($subscription);
    }

    public function update(UpdateSubscriptionRequest $request, Subscription $subscription)
    {
        $canUpdateAny = $request->user()->permitted('subscription.update-any');

        if (!$canUpdateAny) {
            abort(403);
        }

        return new SubscriptionResource(
            $this->subscriptions->saveSubscription($request->all())
        );
    }

    public function updatePayPalIds(Subscription $subscription, Request $request)
    {
        if ($subscription->user_id != $request->user()->id) {
            return abort(403);
        }

        $subscription->paypal_id = $request->paypal_id;

        $subscription->paypal_order_id = $request->paypal_order_id;

        $this->handlePayPalIdUpdate($subscription);

        $subscription->save();

        return new SubscriptionResource($subscription);
    }

    private function handlePayPalIdUpdate(Subscription $subscription)
    {
        if ($subscription->isClean('paypal_id')) return;

        $paypal = new PaypalPaymentGateway(app(CurrencyManager::class));

        if (!$paypal->verifySubscription($subscription)) {
            throw new InvalidArgumentException('paypal_id is invalid');
        }

        event(new SubscriptionVerified($subscription));
    }

    public function listStatuses()
    {
        return SubscriptionStatus::getStatuses();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Subscription  $subscription
     * @return \Illuminate\Http\Response
     */
    public function destroy(Subscription $subscription)
    {
        $subscription->delete();

        return $subscription;
    }
}
