<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreTranslationRequest;
use App\Http\Requests\UpdateTranslationRequest;
use App\Interfaces\FileManager;
use App\Interfaces\MachineTranslation;
use App\Interfaces\TranslationManager;
use App\Jobs\MachineTranslate;
use App\Models\Translation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TranslationController extends Controller
{
    private TranslationManager $translations;

    public function __construct(TranslationManager $translations, MachineTranslation $machineTranslation)
    {
        $this->machineTranslation = $machineTranslation;

        $this->translations = $translations;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate = true;

        if ($request->paginate === 'false') {
            $paginate = false;
        }

        return $this->translations->search(
            paginate: $paginate
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreTranslationRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreTranslationRequest $request)
    {
        return $this->translations->save($request->all());
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Translation  $translation
     * @return \Illuminate\Http\Response
     */
    public function show(Translation $translation)
    {
        return $translation;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateTranslationRequest  $request
     * @param  \App\Models\Translation  $translation
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateTranslationRequest $request, Translation $translation)
    {
        return $this->translations->save($request->all(), $translation);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Translation  $translation
     * @return \Illuminate\Http\Response
     */
    public function destroy(Translation $translation)
    {
        return $this->translations->delete($translation);
    }

    public function activate(Translation $translation)
    {
        return $this->translations->activate($translation);
    }

    public function upload(Translation $translation, FileManager $files, Request $request)
    {
        return $this->translations->upload($request, $translation);
    }

    public function autoTranslate(Translation $translation)
    {
        $completeness = $this->translations->completeness($translation);

        if ($completeness == 100) {
            return [
                'error' => 'Translation is already completed.'
            ];
        }

        dispatch(new MachineTranslate($translation->id));

        return [
            'started' => true
        ];
    }

    public function canAutoTranslate()
    {
        return [
            'enabled' => !empty(config('services.google.api_key'))
        ];
    }
}
