<?php

namespace App\Jobs;

use App\Interfaces\CurrencyManager;
use App\Interfaces\PaymentGateway;
use App\Models\Config;
use App\Models\PaypalPaymentGateway;
use App\Models\StripePaymentGatewayModel;
use App\Repositories\StripePaymentGateway;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class WebhooksRegistrar implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private $paymentGateway;

    public function __construct(PaymentGateway $paymentGateway)
    {
        $this->paymentGateway = $paymentGateway;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $this->registerWebhooks();
    }

    private function registerPayPalWebhook()
    {
        if (!PaypalPaymentGateway::isEnabled()) {
            return;
        }

        if (Config::get('webhook.id')) {
            return;
        }

        $webhooks = collect($this->paymentGateway->listWebhooks());

        $wh = $webhooks->first(fn ($w) => $w['url'] === route('webhooks.paypal'));

        if ($wh) {

            Config::set('webhook.id', $wh['id']);

            return;
        }

        $this->paymentGateway->registerWebhook();
    }

    private function registerStripeWebhook()
    {
        if (Config::get('stripe_webhook.id')) {
            return;
        }

        $stripe = new StripePaymentGateway(app(CurrencyManager::class));

        if (!StripePaymentGatewayModel::isEnabled()) return;

        $webhooks = $stripe->listWebhooks();

        $webhooks = collect($webhooks);

        $wh = $webhooks->first(fn ($w) => $w->url === route('webhooks.stripe'));

        if ($wh) {

            Config::set('stripe_webhook.id', $wh->id);

            return;
        }

        $webhook = $stripe->registerWebhook();

        if ($webhook) {

            Config::set('stripe_webhook.secret', $webhook->secret);
        }
    }

    private function registerWebhooks()
    {
        // On local host, webhooks cannot be registered.
        if (preg_match('/localhost/', url('/'))) {
            return;
        }

        $this->registerPayPalWebhook();

        $this->registerStripeWebhook();
    }
}
