<?php

namespace App\Models;

use App\Events\SavingSubscriptionPlan;
use App\Events\SubscriptionPlanDeleted;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * @property string name
 * @property string frequency
 * @property float price
 * @property float monthly_price
 * @property int number_of_dynamic_qrcodes
 * @property int number_of_scans
 * @property bool is_popular
 * @property bool is_hidden
 * @property bool is_trial
 * @property int trial_days
 * @property array qr_types Allowed types by this plan
 * @property array features
 * @property string paypal_plan_id
 * @property string paypal_product_id
 * @property string stripe_price_id
 * @property string stripe_product_id
 */
class SubscriptionPlan extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * Available values: DAY WEEK MONTH YEAR
     */
    const PAYPAL_YEARLY_INTERVAL = 'YEAR';

    const FREQUENCY_ONE_TIME = 'ONE_TIME';

    private $payment_fields = [
        'paypal_plan_id',
        'paypal_product_id',
        'stripe_price_id',
        'stripe_product_id'
    ];

    protected $fillable = [
        'name',
        'price',
        'frequency',
        'monthly_price',
        'number_of_dynamic_qrcodes',
        'number_of_scans',
        'is_popular',
        'is_hidden',
        'is_trial',
        'trial_days',
        'qr_types',
        'features'
    ];

    protected $appends = ['yearly_price'];

    protected $casts = [
        'is_popular' => 'boolean',
        'is_default_plan' => 'boolean',
        'is_hidden' => 'boolean',
        'is_trial' => 'boolean',
        'qr_types' => 'array',
        'features' => 'array',
    ];

    protected $dispatchesEvents = [
        'saving' => SavingSubscriptionPlan::class,
        'deleted' => SubscriptionPlanDeleted::class
    ];

    public static function boot()
    {
        parent::boot();

        static::saving(function (SubscriptionPlan $model) {
            $model->qr_types = $model->qr_types ?? [];
            $model->features = $model->features ?? [];
        });
    }

    public function yearlyPrice(): Attribute
    {
        return new Attribute(fn () => bcmul(12, $this->monthly_price, 1));
    }

    public function description(): Attribute
    {
        return new Attribute(fn () => implode(', ', [
            $this->name . ' Plan',
            $this->number_of_dynamic_qrcodes . ' Dynamic QR Codes.',
            $this->number_of_scans . ' Scans',
            'Single user'
        ]));
    }



    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    public function exiprationDays()
    {
        if ($this->frequency === $this::FREQUENCY_ONE_TIME) {
            return PHP_FLOAT_MAX;
        }

        if ($this->is_trial) {
            return $this->trial_days;
        }

        return 365;
    }

    public function resetPaymentGatewayFields()
    {
        foreach ($this->payment_fields as $field) {
            $this->{$field} = null;
        }
    }
}
