<?php

namespace App\Repositories;

use App\Http\Requests\StoreBlogPostRequest;
use App\Http\Requests\UpdateBlogPostRequest;
use App\Interfaces\BlogPostManager as BlogPostManagerInterface;
use App\Models\BlogPost;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class BlogPostManager implements BlogPostManagerInterface
{
    private ModelSearchBuilder $search;

    public function __construct(ModelSearchBuilder $search)
    {
        $this->search = $search;
    }

    public function search(Request $request)
    {
        $search = $this->search
            ->init(BlogPost::class, $request, orderByIdOnPaginate: false)
            ->inColumn('title')
            ->search();

        if (!$request->user()?->permitted('blog-post.list-all')) {

            $search->withQuery(function ($query) {
                $query->published();
                $query->orderBy('published_at', 'desc');
            });
        }

        return $search->paginate();
    }

    public function recentPosts(Request $request, int $number = 5)
    {
        return BlogPost::published()
            ->orderBy('published_at', 'desc')
            ->orderBy('id', 'desc')
            ->take($number)
            ->get();
    }

    public function getPost(BlogPost $post, Request $request)
    {
        if ($request->user()?->permitted('blog-post.show-any')) {
            return $post;
        }

        $post = BlogPost::published()->find($post->id);

        if (empty($post)) {
            abort(404);
        }

        return $post;
    }

    public function store(StoreBlogPostRequest $request)
    {
        $post = new BlogPost($request->all());

        $post->user_id = $request->user()->id;

        $post->save();

        return $post;
    }

    public function update(BlogPost $post, UpdateBlogPostRequest $request)
    {
        $post->fill($request->all());

        $post->save();

        return $post;
    }

    public function delete(BlogPost $post)
    {
        $post->delete();

        return $post;
    }

    private function makeView(string $view, $data = [], Request $request)
    {
        $recentPosts = $this->recentPosts($request);

        return view($view, array_merge($data, compact('recentPosts')));
    }

    public function viewPost(string $view, BlogPost $post, Request $request)
    {
        $post = $this->getPost($post, $request);

        return $this->makeView($view, compact('post'), $request);
    }

    public function viewArchive(string $view, Request $request)
    {
        $posts = $this->search($request);

        return $this->makeView($view, compact('posts'), $request);
    }
}
