<?php

namespace App\Repositories;

use App\Interfaces\CurrencyManager;
use Carbon\Carbon;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Interfaces\PaymentGateway;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

use App\Models\PaypalPaymentGateway as PaypalModel;

use Exception;
use Illuminate\Support\Facades\Config;

class PaypalPaymentGateway implements PaymentGateway
{
	private $currencies;

	function __construct(CurrencyManager $currencies)
	{
		$this->currencies = $currencies;
	}

	private function shouldRun()
	{
		$shouldRun = PaypalModel::isEnabled() && config('paypal.client_id') && config('paypal.client_secret');

		return $shouldRun;
	}

	public function getAccessToken()
	{
		if (!$this->shouldRun()) return;

		$request = $this->makeRequest();

		$response = $request

			->withBasicAuth(
				config('paypal.client_id'),
				config('paypal.client_secret')
			)

			->asForm()

			->post('v1/oauth2/token', [
				'grant_type' => 'client_credentials'
			]);

		$token = @$response['access_token'];

		if (!$token) {
			Log::error('Could not get PayPal access token, got ', $response->json());
		}

		return $token;
	}

	private function getProductId($subscriptionPlan)
	{
		if (!$this->shouldRun()) return;

		$route = 'v1/catalogs/products';

		$data = [
			'name' => $subscriptionPlan->name,
			'description' => $subscriptionPlan->description,
			"type" => "DIGITAL",
			"category" => "SOFTWARE"
		];

		$id = $subscriptionPlan->paypal_product_id;

		if (empty($id)) {

			$id = $this->makeApiRequest()->post($route, $data)['id'];

			$subscriptionPlan->paypal_product_id = $id;
		}

		return $id;
	}


	public function verifySubscription(Subscription $subscription)
	{
		if (!$this->shouldRun()) return;

		$result = $this
			->makeApiRequest()
			->get('/v1/billing/subscriptions/' . $subscription->paypal_id);

		return $subscription->paypal_id === @$result['id'];
	}

	public function saveSubscriptionPlan(SubscriptionPlan $subscriptionPlan, bool $forceSync = false)
	{
		if (!$this->shouldRun()) return;

		$product_id = $this->getProductId($subscriptionPlan);

		$currentPlanId = $subscriptionPlan->paypal_plan_id;

		if (!empty($currentPlanId)) {
			$this->deactivatePlan($subscriptionPlan);
		}

		$route = 'v1/billing/plans';

		$data  = [
			"product_id" => $product_id,
			"name" => $subscriptionPlan->name,
			"description" => $subscriptionPlan->description,
			"status" => "ACTIVE",
			"billing_cycles" => [
				[
					"frequency" => [
						"interval_unit" => SubscriptionPlan::PAYPAL_YEARLY_INTERVAL,
						"interval_count" => 1
					],
					"tenure_type" => "REGULAR",
					"sequence" => 1,
					"total_cycles" => 0,
					"pricing_scheme" => [
						"fixed_price" => [
							"value" => $subscriptionPlan->yearly_price,
							"currency_code" => $this->currencies->enabledCurrency()->currency_code
						]
					]
				]
			],
			"payment_preferences" => [
				"auto_bill_outstanding" => true,
				"payment_failure_threshold" => 3
			]
		];

		$result = $this->makeApiRequest()->post($route, $data);

		if (empty($result['id'])) {
			throw new Exception('Expected PayPal response with plan id, got ' . var_export($result, true));
		}

		$subscriptionPlan->paypal_plan_id = $result['id'];
	}

	public function deactivatePlan($subscriptionPlan)
	{
		if (!$this->shouldRun()) return;

		$id = $subscriptionPlan->paypal_plan_id;

		if (empty($id)) return; // no further action is needed if we have no plan id

		$route = "/v1/billing/plans/$id/deactivate";

		$this->makeApiRequest()->post($route);
	}

	public function listTransactions()
	{
		if (!$this->shouldRun()) return;

		$start = Carbon::parse('now')->subMonth(1);

		$end = Carbon::now();

		return $this->makeApiRequest()->get('/v1/reporting/transactions', [
			'start_date' => $start->toIso8601String(),
			'end_date' => $end->toIso8601String()
		])->json();
	}

	/**
	 * Required to be called when app feature is enabled.
	 */
	public function terminateToken()
	{
		if (!$this->shouldRun()) return;

		$token = $this->getAccessToken();

		$this->makeRequest()
			->withBasicAuth(
				config('paypal.client_id'),
				config('paypal.client_secret')
			)->asForm()->acceptJson()
			->post('v1/oauth2/token/terminate', [
				'token' => $token
			]);
	}

	public function registerWebhook()
	{
		if (!$this->shouldRun()) return;

		$route = route('webhooks.paypal');

		return $this->makeApiRequest()->post('v1/notifications/webhooks', [
			'url' => $route,
			'event_types' => [
				['name' => 'PAYMENT.SALE.COMPLETED'],
				['name' => 'PAYMENT.SALE.DENIED'],
				['name' => 'PAYMENT.SALE.PENDING']
			]
		])->json();
	}

	public function clearWebhooks()
	{
		if (!$this->shouldRun()) return;

		$webhooks = $this->listWebhooks();

		foreach ($webhooks as $webhook) {
			$this->deleteWebhook($webhook['id']);
		}

		return $this->listWebhooks();
	}

	public function deleteWebhook($id)
	{
		if (!$this->shouldRun()) return;

		return $this->makeApiRequest()->delete('/v1/notifications/webhooks/' . $id);
	}

	public function listWebhooks()
	{
		if (!$this->shouldRun()) return;

		return $this->makeApiRequest()->get('/v1/notifications/webhooks')->json()['webhooks'];
	}

	public static function boot()
	{
		static::bindConfiguration();
	}

	private static function bindConfiguration()
	{
		$mode = PaypalModel::getMode();

		Config::set('paypal.mode', $mode);

		switch ($mode) {
			case PaypalModel::MODE_SANDBOX:
				Config::set('paypal.endpoint', PaypalModel::ENDPOINT_SANDBOX);
				break;
			case PaypalModel::MODE_LIVE:
				Config::set('paypal.endpoint', PaypalModel::ENDPOINT_LIVE);
				break;
		}

		Config::set('paypal.client_id', PaypalModel::instance()->client_id);

		Config::set('paypal.client_secret', PaypalModel::instance()->client_secret);
	}

	private function makeApiRequest()
	{
		if (!$this->shouldRun()) return;

		$accessToken = $this->getAccessToken();

		return $this->makeRequest()->withToken($accessToken);
	}

	private function makeRequest()
	{
		if (!$this->shouldRun()) return;

		return Http::withHeaders([
			'Accept-Language' => 'en_US'
		])
			->acceptJson()
			->timeout(10)
			->baseUrl(config('paypal.endpoint'));
	}
}
