<?php

namespace App\Repositories;

use App\Interfaces\CurrencyManager;
use App\Interfaces\PaymentGateway;
use App\Models\StripePaymentGatewayModel;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Stripe\StripeClient;
use Throwable;

use App\Models\Config as ConfigModel;

class StripePaymentGateway implements PaymentGateway
{
    private static StripeClient $stripe;

    private CurrencyManager $currencies;

    public function __construct(CurrencyManager $currencies)
    {
        $sk = config('services.stripe.secret_key');

        $this->currencies = $currencies;

        if (empty($this::$stripe) && !empty($sk)) {
            $this::$stripe = new StripeClient($sk);
        }
    }

    public function getAccessToken()
    {
    }

    private function getProductId(SubscriptionPlan $subscriptionPlan)
    {

        if (!$this->shouldRun()) return;

        $data = [
            'name' => $subscriptionPlan->name,
            'description' => $subscriptionPlan->description
        ];

        $productId = $subscriptionPlan->stripe_product_id;


        if (empty($productId)) {

            $product = $this::$stripe->products->create($data);

            $productId = $product->id;

            $subscriptionPlan->stripe_product_id = $productId;
        } else if ($subscriptionPlan->isDirty(['name', 'description']) || $subscriptionPlan->wasChanged(['name', 'description'])) {
            $this::$stripe->products->update($productId, $data);
        }


        return $productId;
    }

    public function saveSubscriptionPlan(SubscriptionPlan $subscriptionPlan, bool $forceSync = false)
    {

        if (!$this->shouldRun()) return;

        $productId = $this->getProductId($subscriptionPlan);

        $priceId = $subscriptionPlan->stripe_price_id;

        $data = [
            'active' => true,
            'unit_amount' => $subscriptionPlan->yearly_price * 100, // cents
            'recurring' => [
                'interval' => 'year',
                'interval_count' => 1,
            ],
            'currency' => $this->currencies->enabledCurrency()->currency_code,
            'product' => $productId
        ];

        if (
            empty($priceId) || $forceSync || $subscriptionPlan->isDirty('monthly_price') || $subscriptionPlan->wasChanged('monthly_price')
        ) {
            $price = $this::$stripe->prices->create($data);

            $subscriptionPlan->stripe_price_id = $price->id;
        }
    }

    public function verifySubscription(Subscription $subscription)
    {
        if (!$this->shouldRun()) return;

        try {
            $subscription = $this::$stripe->subscriptions->retrieve($subscription->stripe_id);

            return $subscription->id == $subscription->stripe_id;
        } catch (\Throwable $th) {
            return false;
        }
    }

    public static function boot()
    {
        static::bindConfiguration();
    }

    private function shouldRun()
    {
        $shouldRun = StripePaymentGatewayModel::isEnabled() && !empty(config('services.stripe.secret_key')) && !empty(config('services.stripe.publisher_key'));

        return $shouldRun;
    }

    private static function bindConfiguration()
    {
        Config::set('services.stripe.publisher_key', StripePaymentGatewayModel::instance()->publisher_key);

        Config::set('services.stripe.secret_key', StripePaymentGatewayModel::instance()->secret_key);
    }

    public function registerWebhook()
    {
        if (!$this->shouldRun()) return;

        return $this::$stripe->webhookEndpoints->create([
            'url' => route('webhooks.stripe'),
            'enabled_events' => [
                'payment_intent.succeeded',
                'payment_intent.payment_failed'
            ],
        ]);
    }

    public function listWebhooks()
    {
        if (!$this->shouldRun()) return [];

        return $this::$stripe->webhookEndpoints->all()->data;
    }

    public function clearWebhooks()
    {
        $webhooks = $this->listWebhooks();

        foreach ($webhooks as $webhook) {
            $this::$stripe->webhookEndpoints->delete($webhook->id);
        }

        ConfigModel::set('stripe_webhook.id', null);

        ConfigModel::set('stripe_webhook.secret', null);
    }

    public function generateCheckoutUrl(Subscription $subscription)
    {
        $session = $this::$stripe->checkout->sessions->create([
            'success_url' => url('/checkout/stripe/success?session_id={CHECKOUT_SESSION_ID}'),
            'cancel_url' => url('/checkout/cancelled'),
            'mode' => 'subscription',
            'client_reference_id' => $subscription->id,
            'line_items' => [
                [
                    'price' => $subscription->subscription_plan->stripe_price_id,
                    'quantity' => 1,
                ]
            ],
            'metadata' => [
                'subscription_id' => $subscription->id
            ],
            'customer_email' => $subscription->user->email
        ]);

        return $session->url;
    }

    public function getCheckoutSessionByPaymentIntent($id)
    {
        return @$this::$stripe->checkout->sessions->all([
            'limit' => 1,
            'payment_intent' => $id
        ])->data[0];
    }

    public function getCheckoutSession($id)
    {
        try {
            return $this::$stripe->checkout->sessions->retrieve($id);
        } catch (\Throwable $th) {
        }

        return null;
    }
}
