<?php

namespace App\Repositories;

use App\Interfaces\QRCodeScanManager;
use App\Interfaces\SubscriptionManager as SubscriptionManagerInterface;

use App\Models\Subscription;
use App\Models\SubscriptionStatus;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;

class SubscriptionManager implements SubscriptionManagerInterface
{
    public function setExpiredSubscriptions()
    {
        $subscriptions = Subscription::with('statuses', 'subscription_plan')->get();

        foreach ($subscriptions as $subscription) {
            $status = $subscription->statuses->first();

            if (!$status) {
                Log::warning(
                    "Subscription without statuses, those are likely test users",
                    [
                        'id' => $subscription->id,
                        'user_id' => $subscription->user_id
                    ]
                );
                continue;
            }

            if ($status->status === SubscriptionStatus::STATUS_EXPIRED) {
                continue;
            }

            $plan = $subscription->subscription_plan;

            if ($status->status === SubscriptionStatus::STATUS_ACTIVE) {

                $planExpirationDays = $plan->exiprationDays();

                $days = Carbon::now()->diffInDays($status->created_at);

                if ($days > $planExpirationDays) {
                    $this->assignExpiredStatus($subscription);
                }
            }
        }
    }

    public function activateSubscription(Subscription $subscription)
    {
        $status = SubscriptionStatus::STATUS_ACTIVE;

        if ($subscription->statuses[0]?->status == $status) {
            return;
        }

        $this->assignStatus($subscription, $status);
    }

    public function deactivateSubscription(Subscription $subscrption)
    {
        if ($subscrption->statuses[0]->status != SubscriptionStatus::STATUS_ACTIVE) {
            return;
        }

        $subscrption->statuses[0]->delete();
    }


    private function assignExpiredStatus($subscription)
    {
        $this->assignStatus($subscription, SubscriptionStatus::STATUS_EXPIRED);
    }

    private function assignStatus($subscription, string $status)
    {
        $found = array_filter(
            SubscriptionStatus::getStatuses(),
            fn ($s) => $s === $status
        );

        if (empty($found)) {
            throw new InvalidArgumentException("Status ($status) is invalid, expected on of: " . implode(', ', SubscriptionStatus::getStatuses()));
        }

        $model = new SubscriptionStatus();

        $model->status = $status;

        $model->subscription_id = $subscription->id;

        $model->save();
    }

    public function saveSubscription($data)
    {
        $subscription = new Subscription();

        if (isset($data['id'])) {
            $subscription = Subscription::findOrFail($data['id']);
        }

        $subscription->fill($data);

        $subscription->save();

        if ($subscription->statuses->first()?->status !== $data['subscription_status']) {
            $this->assignStatus($subscription, $data['subscription_status']);
        }

        $subscription->refresh();

        return $subscription;
    }

    public function userHasActiveSubscription(User $user)
    {
        if ($user->isSuperAdmin()) return true;

        $subscription = $user->subscriptions[0];

        return $subscription->statuses[0]->status === SubscriptionStatus::STATUS_ACTIVE;
    }

    public function userScanLimitReached(User $user)
    {
        if ($user->isSuperAdmin()) return false;

        if (config('app.wplus_integration_enabled')) return false;

        /** @var \App\Interfaces\QRCodeScanManager */
        $scans = app(QRCodeScanManager::class);

        $plan = $user->subscriptions[0]->subscription_plan;

        return $scans->getScanCountOfQrCodesCreatedByUser($user) >= $plan->number_of_scans;
    }
}
