<?php

namespace App\Repositories;

use App\Interfaces\FileManager;
use App\Interfaces\ModelSearchBuilder;
use App\Interfaces\TranslationManager as TranslationManagerInterface;
use App\Models\Translation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;

class TranslationManager implements TranslationManagerInterface
{
    private ModelSearchBuilder $search;

    private FileManager $files;

    private static $activeTranslation;

    public function __construct(ModelSearchBuilder $search, FileManager $files)
    {
        $this->search = $search;

        $this->files = $files;
    }

    public static function t(string $text)
    {
        if (!static::$activeTranslation) {
            try {
                $data = static::loadActiveTranslationFile();

                static::$activeTranslation = json_decode($data, true);
            } catch (\Throwable $th) {
                Log::error($th->getMessage());
            }
        }

        if (isset(static::$activeTranslation[$text]) && !empty(static::$activeTranslation[$text])) {
            return static::$activeTranslation[$text];
        }

        return $text;
    }

    public function search(bool $paginate = true)
    {
        $search = $this->search->init(
            Translation::class,
            request()
        )
            ->inColumn('name')
            ->search();

        if ($paginate) {
            return $search->paginate()
                ->through(function ($row) {
                    $row->completeness = $this->completeness($row);
                    return $row;
                });
        }

        return $search->query()->get();
    }

    public function load(Translation $translation)
    {
        return file_get_contents($this->files->path($translation->file));
    }

    /**
     * Updates the translation file with $data
     */
    public function write($data, Translation $translation)
    {
        file_put_contents(
            $this->files->path($translation->file),
            $data
        );
    }

    public function save($data, ?Translation $translation = null)
    {
        if ($translation?->is_default) return $translation;

        if (!$translation) {
            $translation = new Translation();
        }

        $translation->fill($data);

        $translation->save();

        return $translation;
    }

    public function delete(Translation $translation)
    {
        if ($translation->is_default) return $translation;

        return $translation->delete();
    }

    public function completeness(Translation $translation)
    {
        if (!$translation->file) return 0;

        if ($translation->is_default) return 100;

        $json = $this->load($translation);

        $data = json_decode($json, true);

        $completed = array_reduce(
            $data,
            fn ($count, $value) => empty($value) ? $count : $count + 1,
            0
        );

        return floor($completed / count($data) * 100);
    }

    public function verifyTranslationFile(Translation $translation)
    {
        $json = $this->load($translation);

        $data = json_decode($json, true);

        if (empty($data)) {
            throw new InvalidArgumentException("Translation file has invalid json data.");
        }

        $defaultTranslationJson = $this->loadDefaultTranslation();

        $defaultData = json_decode($defaultTranslationJson, true);

        $translationCount = count(array_keys($data));
        $defaultTranslationCount = count(array_keys($defaultData));

        if ($translationCount !== $defaultTranslationCount) {
            Log::error("Translation file ($translation->locale) has $translationCount keys, while default translation has $defaultTranslationCount");

            throw new InvalidArgumentException("Translation file has invalid keys. ($translation->locale)");
        }

        foreach (array_keys($defaultData) as $key) {
            $found = array_filter(array_keys($data), fn ($dataKey) => $key === $dataKey);

            if (!$found) {
                throw new InvalidArgumentException(
                    "Translation key ($key) was not found."
                );
            }
        }
    }

    private function loadDefaultTranslation()
    {
        $defaultTranslation = Translation::where('is_default', true)->first();

        return $this->load($defaultTranslation);
    }

    public static function loadActiveTranslationFile()
    {
        try {
            $translation = Translation::where('is_active', true)->first();

            if ($translation) {
                return app(static::class)->load($translation);
            }
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            return '{}';
        }

        return '{}';
    }

    public function upload(Request $request, Translation $translation)
    {
        $request->merge([
            'attachable_type' => Translation::class,
            'attachable_id' => $translation->id,
            'type' => FileManager::FILE_TYPE_TRANSLATION
        ]);

        $this->files->setFileValidator(function () use ($translation) {
            $translation->refresh();
            $this->verifyTranslationFile($translation);
        });

        $result = $this->files->store($request);

        return $result;
    }

    public function activate(Translation $translation)
    {
        if (app()->environment() === 'demo') {
            return [
                'error' => 'Cannot activate translation in demo'
            ];
        }

        if (
            !$translation->is_default &&
            $this->completeness($translation) < 100
        ) {
            return [
                'error' => 'Translation must be completed before activation'
            ];
        }

        Translation::all()->each(function ($translation) {
            $translation->is_active = false;
            $translation->save();
        });

        $translation->is_active = true;

        $translation->save();

        return $translation;
    }
}
