<?php

namespace App\Support;

use App\Interfaces\BlogPostManager;
use App\Models\Config;
use App\Models\ContentBlock;
use App\Models\CustomCode;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Throwable;

class ContentManager
{
    private BlogPostManager $posts;

    public function __construct(BlogPostManager $posts)
    {
        $this->posts = $posts;
    }

    public function bodyClass($classes = '')
    {
        $path = Str::slug(request()->path()) ?: 'base';

        return "class='path-$path $classes'";
    }

    public function recentBlogSection()
    {
        try {
            $recentPosts = $this->posts->recentPosts(request(), 6);

            return view('blog/recent-section', compact('recentPosts'));
        } catch (\Throwable $ex) {
        }
    }

    public function contentBlocks($position, $template = 'simple')
    {
        $this->addPositionIfNeeded($position);

        return view('content-block.main', compact('position', 'template'));
    }

    public function customCode($position)
    {
        $this->addPositionIfNeeded($position, 'custom-code-positions');

        $codes = '';

        try {

            $codes = CustomCode::where(
                'position',
                $position
            )
                ->orderBy('sort_order', 'asc')
                ->get()
                ->map(function ($model) {

                    if ($model->language === 'javascript') {
                        return "<script>$model->code</script>";
                    }

                    if ($model->language === 'css') {
                        return "<style>$model->code</style>";
                    }

                    return $model->code;
                })
                ->join("\n\r");
        } catch (Throwable $th) {
            Log::error('Error while rendering custom code. ' . $th->getMessage());
        }


        return $codes;
    }

    public function addPositionIfNeeded($position, $configKey = 'positions')
    {
        $positions = $this->getPositions($configKey);

        $found = array_filter($positions, fn ($p) => $p == $position);

        if (!$found) {
            $positions[] = $position;

            ConfigFileManager::saveJson('content-manager.' . $configKey, $positions);
        }
    }

    private function getPositions($configKey = 'positions')
    {
        $positions = config('content-manager.' . $configKey, '');

        $positions = json_decode($positions) ?: [];

        return $positions;
    }

    public function renderConfigThemeStyles()
    {
        $configKeys = explode(',', 'theme.primary_0,theme.primary_1,theme.accent_0,theme.accent_1,theme.input_placeholder_font_style');

        $cssVar = function ($key) {
            $str = str_replace('theme.', '', $key);
            $str = str_replace('_', '-', $str);
            return "--$str";
        };

        $styles = array_reduce($configKeys, function ($result, $key) use ($cssVar) {
            if (json_decode(config($key))) {
                $result .= sprintf("%s: %s; \n", $cssVar($key), json_decode(config($key)));
            }

            return $result;
        }, '');

        $styles = trim($styles);

        return "<style> :root { $styles } </style>";
    }

    /**
     * Search if there are any blocks in any position that matches $search
     */
    public function hasAnyBlocks($search)
    {
        try {
            $positions = collect($this->getPositions())->filter(
                fn ($p) => preg_match($search, $p)
            );

            $count = ContentBlock::whereIn('position', $positions)->count();

            return $count > 0;
        } catch (\Throwable $th) {
            return false;
        }
    }

    public function renderHomePageMetaDescription()
    {
        if (request()->path() === '/') {
            return sprintf('<meta name="description" content="%s" />', Config::get('homepage.meta_description'));
        }
    }
}
