<?php

namespace App\Support;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

/**
 * Google font variants: 5148
 * Google font count: 1442
 */
class GoogleFonts
{
    private $cacheFs;

    public function __construct()
    {
        $this->cacheFs = Storage::disk('db');
    }

    private function cachePath()
    {
        return 'raw/google_fonts.json';
    }

    public function list()
    {
        $cached = $this->cacheFs->get($this->cachePath());

        $cached = json_decode($cached, true);

        if (empty($cached)) {
            $response = $this->get();
            $cached = $response->body();
            $cached = json_decode($cached, true);

            $this->cacheFs->put($this->cachePath(), json_encode($cached, JSON_PRETTY_PRINT));
        }

        return $cached['items'];
    }

    public function clearFontCache()
    {
        foreach (glob(Storage::path('google_fonts/*.ttf')) as $file) {
            Storage::delete($file);
        }
    }

    public function countVariants()
    {
        $cached = $this->cacheFs->get($this->cachePath());

        $fonts = json_decode($cached, true)['items'];

        return array_reduce($fonts, function ($sum, $font) {
            return $sum + count($font['variants']);
        }, 0);
    }

    public function listFamilies()
    {
        $list = $this->list();

        return array_map(function ($font) {
            $item = [];

            $item['family'] = $font['family'];

            $item['variants'] = $font['variants'];

            return $item;
        }, $list);
    }

    public function details($family)
    {
        return collect($this->list())->first(function ($font) use ($family) {
            return $font['family'] === $family;
        });
    }

    public function getFontFile($family, $variant = 'regular')
    {
        $path = sprintf("google_fonts/%s_%s.ttf", $family, $variant);

        if (Storage::exists($path)) {
            return Storage::path($path);
        }

        $font = $this->details($family);

        $files = $font['files'];

        if (!$variant) {
            // Get thickest variation
            $variant = collect($font['variants'])->reduce(function ($max, $v) {
                if (is_numeric($v)) {
                    $max = max($max, $v);
                }

                return $max;
            }, 0) ?: 'regular';
        }

        $variant_is_found = array_filter($font['variants'], fn ($v) => $v === $variant);

        if (!$variant_is_found) {
            $fallback = 'regular';
            $variant = $fallback;
        }

        $data = file_get_contents($files[$variant]);

        Storage::put($path, $data);

        return Storage::path($path);
    }

    private function get()
    {
        return Http::get(
            'https://www.googleapis.com/webfonts/v1/webfonts?key=' . config('services.google.api_key')
        );
    }
}
